"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toolMatchSelection = exports.isByIdsToolSelection = exports.filterToolsBySelection = exports.allToolsSelectionWildcard = exports.allToolsSelection = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * "all tools" wildcard which can be used for tool selection
 */
const allToolsSelectionWildcard = exports.allToolsSelectionWildcard = '*';

/**
 * Constant tool selection to select all tools
 */
const allToolsSelection = exports.allToolsSelection = [{
  tool_ids: [allToolsSelectionWildcard]
}];

/**
 * Represents a tool selection based on individual tool IDs
 *
 * The '*' wildcard can be used for ID selection, to inform that all tools should be selected.
 *
 * @example
 * ```ts
 * // select all available tools
 * const allTools: ByIdsToolSelection = { tool_ids: ['*'] }
 *
 * // select toolA and toolB
 * const toolAB: ByIdsToolSelection = { tool_ids: ['toolA', 'toolB'] }
 * ```
 */

/**
 * All possible subtypes for tool selection - for now there is only one.
 */

/**
 * Check if a given {@link ToolSelection} is a {@link ByIdsToolSelection}
 */
const isByIdsToolSelection = toolSelection => {
  return 'tool_ids' in toolSelection && Array.isArray(toolSelection.tool_ids);
};

/**
 * Returns all tools matching at least one of the provided tool selections.
 */
exports.isByIdsToolSelection = isByIdsToolSelection;
const filterToolsBySelection = (tools, toolSelection) => {
  return tools.filter(tool => toolSelection.some(selection => toolMatchSelection(tool, selection)));
};

/**
 * Returns true if the given tool descriptor matches the provided tool selection.
 */
exports.filterToolsBySelection = filterToolsBySelection;
const toolMatchSelection = (tool, toolSelection) => {
  if (toolSelection.tool_ids.includes(allToolsSelectionWildcard)) {
    return true;
  }
  return toolSelection.tool_ids.includes(tool.id);
};
exports.toolMatchSelection = toolMatchSelection;