"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchAnalyticsMapData = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _std = require("@kbn/std");
var _lodash = require("lodash");
var _mlDataFrameAnalyticsUtils = require("@kbn/ml-data-frame-analytics-utils");
var _kibana = require("../../../contexts/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useFetchAnalyticsMapData = () => {
  const mlApi = (0, _kibana.useMlApi)();
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const [elements, setElements] = (0, _react.useState)([]);
  const [error, setError] = (0, _react.useState)();
  const [message, setMessage] = (0, _react.useState)();
  // Keeps track of which nodes have been used as root so we can refetch related nodes on refresh
  const [usedAsRoot, setUsedAsRoot] = (0, _react.useState)({});
  const nodeDetails = (0, _react.useRef)({});
  const fetchAndSetElements = async (idToUse, treatAsRoot, type) => {
    setIsLoading(true);
    if (treatAsRoot && usedAsRoot[idToUse] === undefined) {
      setUsedAsRoot({
        ...usedAsRoot,
        [idToUse]: type
      });
    }
    // Pass in treatAsRoot flag - endpoint will take job or index to grab jobs created from it
    const analyticsMap = await mlApi.dataFrameAnalytics.getDataFrameAnalyticsMap(idToUse, treatAsRoot, type);
    const {
      elements: nodeElements,
      details,
      error: fetchError
    } = analyticsMap;
    if (fetchError !== null) {
      setIsLoading(false);
      setError(fetchError);
    }
    if ((nodeElements === null || nodeElements === void 0 ? void 0 : nodeElements.length) === 0) {
      setMessage(_i18n.i18n.translate('xpack.ml.dataframe.analyticsMap.emptyResponseMessage', {
        defaultMessage: 'No related analytics jobs found for {id}.',
        values: {
          id: idToUse
        }
      }));
    }
    if ((nodeElements === null || nodeElements === void 0 ? void 0 : nodeElements.length) > 0) {
      if (treatAsRoot === false) {
        setElements(nodeElements);
        nodeDetails.current = details;
      } else {
        const uniqueElements = (0, _lodash.uniqWith)([...nodeElements, ...elements], _lodash.isEqual);
        setElements(uniqueElements);
        nodeDetails.current = {
          ...details,
          ...nodeDetails.current
        };
      }
    }
    setIsLoading(false);
  };
  const fetchAndSetElementsWrapper = async params => {
    const {
      analyticsId,
      id,
      modelId,
      type
    } = params !== null && params !== void 0 ? params : {};
    const treatAsRoot = id !== undefined;
    let idToUse;
    if (id !== undefined) {
      idToUse = id;
    } else if (modelId !== undefined) {
      idToUse = modelId;
    } else {
      idToUse = analyticsId;
    }
    await fetchAndSetElements(idToUse, treatAsRoot, modelId !== undefined && treatAsRoot === false ? _mlDataFrameAnalyticsUtils.JOB_MAP_NODE_TYPES.TRAINED_MODEL : type);

    // If related nodes had been fetched from any node then refetch
    if (Object.keys(usedAsRoot).length) {
      await (0, _std.asyncForEach)(Object.keys(usedAsRoot), async nodeId => {
        await fetchAndSetElements(nodeId, true, usedAsRoot[nodeId]);
      });
    }
  };
  return {
    elements,
    error,
    fetchAndSetElementsWrapper,
    isLoading,
    message,
    nodeDetails: nodeDetails.current,
    setElements,
    setError
  };
};
exports.useFetchAnalyticsMapData = useFetchAnalyticsMapData;