"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.criticalTableLookup = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Performs a lookup in a critical values table to determine the significance level
 * associated with a given chi-squared statistic and degrees of freedom.
 *
 * @param {number} chi2Statistic - The chi-squared statistic for which the significance level is to be determined.
 * @param {number} df - The degrees of freedom (an integer) for the chi-squared test.
 * @returns {number} The significance level corresponding to the chi-squared statistic and degrees of freedom.
 * @throws {Error} If df is less than 1 or not an integer.
 */
const criticalTableLookup = (chi2Statistic, df) => {
  if (df < 1) return 1;
  if (!Number.isInteger(df)) throw Error('Degrees of freedom must be a valid integer');

  // Get the row index
  const rowIndex = df - 1;

  // Get the column index
  let minDiff = Math.abs(_constants.CRITICAL_VALUES_TABLE[rowIndex][0] - chi2Statistic);
  let columnIndex = 0;
  for (let j = 1; j < _constants.CRITICAL_VALUES_TABLE[rowIndex].length; j++) {
    const diff = Math.abs(_constants.CRITICAL_VALUES_TABLE[rowIndex][j] - chi2Statistic);
    if (diff < minDiff) {
      minDiff = diff;
      columnIndex = j;
    }
  }

  // Determine the significance level from the column index
  const significanceLevel = _constants.SIGNIFICANCE_LEVELS[columnIndex];
  return significanceLevel;
};
exports.criticalTableLookup = criticalTableLookup;