"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSubscription = exports.useOperator = exports.useObservableState = exports.useObservable = exports.useLatest = exports.useBehaviorSubject = exports.tapUnsubscribe = void 0;
var _react = require("react");
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useLatest = value => {
  const valueRef = (0, _react.useRef)(value);
  valueRef.current = value;
  return valueRef;
};
exports.useLatest = useLatest;
const useObservable = (createObservableOnce, inputValues) => {
  const [output$, next] = useBehaviorSubject(createObservableOnce, () => inputValues);
  (0, _react.useEffect)(() => {
    next(inputValues);
    // `inputValues` can't be statically analyzed
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, inputValues);
  return output$;
};
exports.useObservable = useObservable;
const useBehaviorSubject = (deriveObservableOnce, createInitialValue) => {
  const [[subject$, next], _] = (0, _react.useState)(() => {
    const newSubject$ = new _rxjs.BehaviorSubject(createInitialValue());
    const newNext = newSubject$.next.bind(newSubject$);
    return [newSubject$, newNext];
  });
  const [output$] = (0, _react.useState)(() => deriveObservableOnce(subject$));
  return [output$, next];
};
exports.useBehaviorSubject = useBehaviorSubject;
const useObservableState = (state$, initialState) => {
  const [latestValue, setLatestValue] = (0, _react.useState)(initialState);
  const [latestError, setLatestError] = (0, _react.useState)();
  useSubscription(state$, {
    next: setLatestValue,
    error: setLatestError
  });
  return {
    latestValue,
    latestError
  };
};
exports.useObservableState = useObservableState;
const useSubscription = (input$, {
  next,
  error,
  complete,
  unsubscribe
}) => {
  const latestSubscription = (0, _react.useRef)();
  const latestNext = useLatest(next);
  const latestError = useLatest(error);
  const latestComplete = useLatest(complete);
  const latestUnsubscribe = useLatest(unsubscribe);
  (0, _react.useEffect)(() => {
    const fixedUnsubscribe = latestUnsubscribe.current;
    const subscription = input$.subscribe({
      next: value => {
        var _latestNext$current;
        return (_latestNext$current = latestNext.current) === null || _latestNext$current === void 0 ? void 0 : _latestNext$current.call(latestNext, value);
      },
      error: value => {
        var _latestError$current;
        return (_latestError$current = latestError.current) === null || _latestError$current === void 0 ? void 0 : _latestError$current.call(latestError, value);
      },
      complete: () => {
        var _latestComplete$curre;
        return (_latestComplete$curre = latestComplete.current) === null || _latestComplete$curre === void 0 ? void 0 : _latestComplete$curre.call(latestComplete);
      }
    });
    latestSubscription.current = subscription;
    return () => {
      subscription.unsubscribe();
      fixedUnsubscribe === null || fixedUnsubscribe === void 0 ? void 0 : fixedUnsubscribe();
    };
  }, [input$, latestNext, latestError, latestComplete, latestUnsubscribe]);
  return latestSubscription.current;
};
exports.useSubscription = useSubscription;
const useOperator = (input$, operator) => {
  const latestOperator = useLatest(operator);
  return useObservable(inputs$ => inputs$.pipe((0, _rxjs.switchMap)(([currentInput$]) => latestOperator.current(currentInput$))), [input$]);
};
exports.useOperator = useOperator;
const tapUnsubscribe = onUnsubscribe => source$ => {
  return new _rxjs.Observable(subscriber => {
    const subscription = source$.subscribe({
      next: value => subscriber.next(value),
      error: error => subscriber.error(error),
      complete: () => subscriber.complete()
    });
    return () => {
      onUnsubscribe();
      subscription.unsubscribe();
    };
  });
};
exports.tapUnsubscribe = tapUnsubscribe;