"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LinksStorage = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _common = require("../../common");
var _cm_services = require("./schema/cm_services");
var _latest = require("./schema/latest");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const savedObjectClientFromRequest = async ctx => {
  if (!ctx.requestHandlerContext) {
    throw new Error('Storage context.requestHandlerContext missing.');
  }
  const {
    savedObjects
  } = await ctx.requestHandlerContext.core;
  return savedObjects.client;
};
const searchArgsToSOFindOptions = (query, options) => {
  return {
    type: _common.LINKS_SAVED_OBJECT_TYPE,
    searchFields: options !== null && options !== void 0 && options.onlyTitle ? ['title'] : ['title^3', 'description'],
    search: query.text,
    perPage: query.limit,
    page: query.cursor ? +query.cursor : undefined,
    defaultSearchOperator: 'AND'
  };
};
class LinksStorage {
  constructor({
    logger,
    throwOnResultValidationError
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "throwOnResultValidationError", void 0);
    (0, _defineProperty2.default)(this, "mSearch", {
      savedObjectType: _common.LINKS_SAVED_OBJECT_TYPE,
      toItemResult: (ctx, savedObject) => {
        const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
        const contentItem = (0, _latest.savedObjectToItem)(savedObject);
        const validationError = transforms.mSearch.out.result.validate(contentItem);
        if (validationError) {
          if (this.throwOnResultValidationError) {
            throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
          } else {
            this.logger.warn(`Invalid response. ${validationError.message}`);
          }
        }

        // Validate DB response and DOWN transform to the request version
        const {
          value,
          error: resultError
        } = transforms.mSearch.out.result.down(
        // @ts-expect-error - fix type error
        contentItem, undefined,
        // do not override version
        {
          validate: false
        } // validation is done above
        );
        if (resultError) {
          throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
        }
        return value;
      }
    });
    this.logger = logger;
    this.throwOnResultValidationError = throwOnResultValidationError !== null && throwOnResultValidationError !== void 0 ? throwOnResultValidationError : false;
  }
  async get(ctx, id) {
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);

    // Save data in DB
    const {
      saved_object: savedObject,
      alias_purpose: aliasPurpose,
      alias_target_id: aliasTargetId,
      outcome
    } = await soClient.resolve(_common.LINKS_SAVED_OBJECT_TYPE, id);
    const item = (0, _latest.savedObjectToItem)(savedObject);
    const response = {
      item,
      meta: {
        aliasPurpose,
        aliasTargetId,
        outcome
      }
    };
    const validationError = transforms.get.out.result.validate(response);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.get.out.result.down(
    // @ts-expect-error - fix type error
    response, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async bulkGet() {
    // Not implemented
    throw new Error(`[bulkGet] has not been implemented. See LinksStorage class.`);
  }
  async create(ctx, data, options) {
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);

    // Validate input (data & options) & UP transform them to the latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.create.in.data.up(data);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.create.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const {
      attributes,
      references
    } = (0, _latest.itemToAttributes)(dataToLatest);

    // Save data in DB
    const savedObject = await soClient.create(_common.LINKS_SAVED_OBJECT_TYPE, attributes, {
      ...optionsToLatest,
      references
    });
    const item = (0, _latest.savedObjectToItem)(savedObject);
    const validationError = transforms.create.out.result.validate({
      item
    });
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate DB response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.create.out.result.down(
    // @ts-expect-error - fix type error
    {
      item
    }, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async update(ctx, id, data, options) {
    var _ref;
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);

    // Validate input (data & options) & UP transform them to the latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.update.in.data.up(data);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.update.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const {
      attributes,
      references
    } = (0, _latest.itemToAttributes)(dataToLatest);

    // Save data in DB
    const partialSavedObject = await soClient.update(_common.LINKS_SAVED_OBJECT_TYPE, id, attributes, {
      references: [...references, ...((_ref = optionsToLatest.references) !== null && _ref !== void 0 ? _ref : [])]
    });
    const item = (0, _latest.savedObjectToItem)(partialSavedObject);
    const validationError = transforms.update.out.result.validate({
      item
    });
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate DB response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.update.out.result.down(
    // @ts-expect-error - fix type error
    {
      item
    }, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async delete(ctx, id,
  // force is necessary to delete saved objects that exist in multiple namespaces
  options) {
    var _options$force;
    const soClient = await savedObjectClientFromRequest(ctx);
    await soClient.delete(_common.LINKS_SAVED_OBJECT_TYPE, id, {
      force: (_options$force = options === null || options === void 0 ? void 0 : options.force) !== null && _options$force !== void 0 ? _options$force : false
    });
    return {
      success: true
    };
  }
  async search(ctx, query, options) {
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);

    // Validate and UP transform the options
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.search.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid payload. ${optionsError.message}`);
    }
    const soQuery = searchArgsToSOFindOptions(query, optionsToLatest);
    // Execute the query in the DB
    const soResponse = await soClient.find(soQuery);
    const hits = await Promise.all(soResponse.saved_objects.map(async so => {
      const item = (0, _latest.savedObjectToItem)(so);
      return item;
    })
    // Ignore any saved objects that failed to convert to items.
    .filter(item => item !== null));
    const response = {
      hits,
      pagination: {
        total: soResponse.total
      }
    };
    const validationError = transforms.search.out.result.validate(response);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate the response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.search.out.result.down(
    // @ts-expect-error - fix type error
    response, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
}
exports.LinksStorage = LinksStorage;