"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.injectESQLQueryIntoLensLayers = void 0;
exports.mergeSuggestionWithVisContext = mergeSuggestionWithVisContext;
exports.switchVisualizationType = switchVisualizationType;
var _visualizationUtils = require("@kbn/visualization-utils");
var _esqlUtils = require("@kbn/esql-utils");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const datasourceHasIndexPatternRefs = unknownDatasource => {
  return Boolean(unknownDatasource && (unknownDatasource === null || unknownDatasource === void 0 ? void 0 : unknownDatasource.indexPatternRefs) !== undefined);
};

/**
 * Injects the ESQL query into the lens layers. This is used to keep the query in sync with the lens layers.
 * @param attributes, the current lens attributes
 * @param query, the new query to inject
 * @returns the new lens attributes with the query injected
 */
const injectESQLQueryIntoLensLayers = (attributes, query, suggestion) => {
  const datasourceId = (0, _visualizationUtils.getDatasourceId)(attributes.state.datasourceStates);

  // if the datasource is formBased, we should not fix the query
  if (!datasourceId || datasourceId === 'formBased') {
    return attributes;
  }
  if (!attributes.state.datasourceStates[datasourceId]) {
    return attributes;
  }
  const indexPattern = (0, _esqlUtils.getIndexPatternFromESQLQuery)(query.esql);

  // Find matching index pattern reference from suggestion
  let indexPatternRef;
  if (datasourceHasIndexPatternRefs(suggestion.datasourceState)) {
    const suggestionRefs = suggestion.datasourceState.indexPatternRefs;
    indexPatternRef = suggestionRefs === null || suggestionRefs === void 0 ? void 0 : suggestionRefs.find(ref => ref.title === indexPattern);
  }
  const datasourceState = structuredClone(attributes.state.datasourceStates[datasourceId]);

  // Update each layer with the new query and index pattern if needed
  if (datasourceState !== null && datasourceState !== void 0 && datasourceState.layers) {
    Object.values(datasourceState.layers).forEach(layer => {
      if (!(0, _lodash.isEqual)(layer.query, query)) {
        var _indexPatternRef$id, _indexPatternRef;
        layer.query = query;
        const index = (_indexPatternRef$id = (_indexPatternRef = indexPatternRef) === null || _indexPatternRef === void 0 ? void 0 : _indexPatternRef.id) !== null && _indexPatternRef$id !== void 0 ? _indexPatternRef$id : layer.index;
        if (index) {
          layer.index = index;
        }
      }
    });
  }
  return {
    ...attributes,
    state: {
      ...attributes.state,
      datasourceStates: {
        ...attributes.state.datasourceStates,
        [datasourceId]: datasourceState
      }
    }
  };
};

/**
 * Returns the suggestion updated with external visualization state for ES|QL charts
 * The visualization state is merged with the suggestion if the datasource is textBased, the columns match the context and the visualization type matches
 * @param suggestion the suggestion to be updated
 * @param visAttributes the preferred visualization attributes
 * @param context the lens suggestions api context as being set by the consumers
 * @returns updated suggestion
 */
exports.injectESQLQueryIntoLensLayers = injectESQLQueryIntoLensLayers;
function mergeSuggestionWithVisContext({
  suggestion,
  visAttributes,
  context
}) {
  if (!('textBasedColumns' in context)) {
    return suggestion;
  }

  // it should be one of 'formBased'/'textBased' and have value
  const datasourceId = (0, _visualizationUtils.getDatasourceId)(visAttributes.state.datasourceStates);

  // if the datasource is formBased, we should not merge
  if (!datasourceId || datasourceId === 'formBased') {
    return suggestion;
  }
  const datasourceState = Object.assign({}, visAttributes.state.datasourceStates[datasourceId]);

  // Verify that all layer columns exist in the query result
  if (!(datasourceState !== null && datasourceState !== void 0 && datasourceState.layers)) {
    return suggestion;
  }
  const hasInvalidColumns = Object.values(datasourceState.layers).some(layer => {
    var _layer$columns;
    return (_layer$columns = layer.columns) === null || _layer$columns === void 0 ? void 0 : _layer$columns.some(column => {
      var _context$textBasedCol;
      return !(context !== null && context !== void 0 && (_context$textBasedCol = context.textBasedColumns) !== null && _context$textBasedCol !== void 0 && _context$textBasedCol.find(contextCol => contextCol.id === column.fieldName));
    });
  });
  if (hasInvalidColumns) {
    return suggestion;
  }
  const layerIds = Object.keys(datasourceState.layers);

  // Update attributes with current query if available
  const updatedVisAttributes = context && 'query' in context && context.query ? injectESQLQueryIntoLensLayers(visAttributes, context.query, suggestion) : visAttributes;
  try {
    return {
      title: updatedVisAttributes.title,
      visualizationId: updatedVisAttributes.visualizationType,
      visualizationState: updatedVisAttributes.state.visualization,
      keptLayerIds: layerIds,
      datasourceState: updatedVisAttributes.state.datasourceStates[datasourceId],
      datasourceId,
      columns: suggestion.columns,
      changeType: suggestion.changeType,
      score: suggestion.score,
      previewIcon: suggestion.previewIcon
    };
  } catch {
    return suggestion;
  }
}

/**
 * Switches the visualization type of a suggestion to the specified visualization type
 * @param visualizationMap the visualization map
 * @param targetTypeId the target visualization type to switch to
 * @param familyType the family type of the current suggestion
 * @param shouldSwitch whether the visualization type should be switched
 * @returns updated suggestion or undefined if no switch was made
 */
function switchVisualizationType({
  visualizationMap,
  suggestions,
  targetTypeId,
  familyType,
  forceSwitch
}) {
  var _visualizationInstanc;
  const suggestion = suggestions.find(s => s.visualizationId === familyType);
  const visualizationInstance = visualizationMap[familyType];
  const currentTypeId = suggestion && targetTypeId && (visualizationInstance === null || visualizationInstance === void 0 ? void 0 : visualizationInstance.getVisualizationTypeId(suggestion.visualizationState));

  // Determine if a switch is required either
  // via force flag
  // or by checking if the target type is supported by the family chart type
  const shouldSwitch = forceSwitch || targetTypeId && (visualizationInstance === null || visualizationInstance === void 0 ? void 0 : (_visualizationInstanc = visualizationInstance.isSubtypeSupported) === null || _visualizationInstanc === void 0 ? void 0 : _visualizationInstanc.call(visualizationInstance, targetTypeId)) && currentTypeId !== targetTypeId;
  if (shouldSwitch && suggestion && familyType && targetTypeId) {
    var _visualizationInstanc2;
    const visualizationState = visualizationInstance === null || visualizationInstance === void 0 ? void 0 : (_visualizationInstanc2 = visualizationInstance.switchVisualizationType) === null || _visualizationInstanc2 === void 0 ? void 0 : _visualizationInstanc2.call(visualizationInstance, targetTypeId, suggestion === null || suggestion === void 0 ? void 0 : suggestion.visualizationState);
    return [{
      ...suggestion,
      visualizationState
    }];
  }
}