"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkGetFleetProxies = bulkGetFleetProxies;
exports.createFleetProxy = createFleetProxy;
exports.deleteFleetProxy = deleteFleetProxy;
exports.getFleetProxy = getFleetProxy;
exports.getFleetProxyRelatedSavedObjects = getFleetProxyRelatedSavedObjects;
exports.listFleetProxies = listFleetProxies;
exports.updateFleetProxy = updateFleetProxy;
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _constants = require("../constants");
var _errors = require("../errors");
var _app_context = require("./app_context");
var _fleet_server_host = require("./fleet_server_host");
var _output = require("./output");
var _download_source = require("./download_source");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function savedObjectToFleetProxy(so) {
  const {
    proxy_headers: proxyHeaders,
    ...rest
  } = so.attributes;
  return {
    id: so.id,
    proxy_headers: proxyHeaders && proxyHeaders !== '' ? JSON.parse(proxyHeaders) : undefined,
    ...rest
  };
}
function fleetProxyDataToSOAttribute(data) {
  const {
    proxy_headers: proxyHeaders,
    ...rest
  } = data;
  return {
    proxy_headers: proxyHeaders ? JSON.stringify(proxyHeaders) : null,
    ...rest
  };
}
async function listFleetProxies(soClient) {
  const res = await soClient.find({
    type: _constants.FLEET_PROXY_SAVED_OBJECT_TYPE,
    perPage: _constants.SO_SEARCH_LIMIT
  });
  return {
    items: res.saved_objects.map(savedObjectToFleetProxy),
    total: res.total,
    page: res.page,
    perPage: res.per_page
  };
}
async function createFleetProxy(soClient, data, options) {
  const logger = _app_context.appContextService.getLogger();
  logger.debug(`Creating fleet proxy ${data}`);
  const res = await soClient.create(_constants.FLEET_PROXY_SAVED_OBJECT_TYPE, fleetProxyDataToSOAttribute(data), {
    id: options === null || options === void 0 ? void 0 : options.id,
    overwrite: options === null || options === void 0 ? void 0 : options.overwrite
  });
  logger.debug(`Created fleet proxy ${options === null || options === void 0 ? void 0 : options.id}`);
  return savedObjectToFleetProxy(res);
}
async function getFleetProxy(soClient, id) {
  const res = await soClient.get(_constants.FLEET_PROXY_SAVED_OBJECT_TYPE, id);
  return savedObjectToFleetProxy(res);
}
async function deleteFleetProxy(soClient, esClient, id, options) {
  const logger = _app_context.appContextService.getLogger();
  logger.debug(`Deleting fleet proxy ${id}`);
  const fleetProxy = await getFleetProxy(soClient, id);
  if (fleetProxy.is_preconfigured && !(options !== null && options !== void 0 && options.fromPreconfiguration)) {
    throw new _errors.FleetProxyUnauthorizedError(`Cannot delete ${id} preconfigured proxy`);
  }
  const {
    outputs,
    fleetServerHosts,
    downloadSources
  } = await getFleetProxyRelatedSavedObjects(soClient, id);
  if (
  // download sources cannot be preconfigured
  [...fleetServerHosts, ...outputs].some(fleetServerHostOrOutput => fleetServerHostOrOutput.is_preconfigured) && !(options !== null && options !== void 0 && options.fromPreconfiguration)) {
    throw new _errors.FleetProxyUnauthorizedError('Cannot delete a proxy used in a preconfigured fleet server hosts or output.');
  }
  await updateRelatedSavedObject(soClient, esClient, fleetServerHosts, outputs, downloadSources);
  logger.debug(`Deleted fleet proxy ${id}`);
  return await soClient.delete(_constants.FLEET_PROXY_SAVED_OBJECT_TYPE, id);
}
async function updateFleetProxy(soClient, id, data, options) {
  const logger = _app_context.appContextService.getLogger();
  logger.debug(`Updating fleet proxy ${id}`);
  const originalItem = await getFleetProxy(soClient, id);
  if (data.is_preconfigured && !(options !== null && options !== void 0 && options.fromPreconfiguration)) {
    throw new _errors.FleetProxyUnauthorizedError(`Cannot update ${id} preconfigured proxy`);
  }
  await soClient.update(_constants.FLEET_PROXY_SAVED_OBJECT_TYPE, id, fleetProxyDataToSOAttribute(data));
  logger.debug(`Updated fleet proxy ${id}`);
  return {
    ...originalItem,
    ...data
  };
}
async function bulkGetFleetProxies(soClient, ids, {
  ignoreNotFound = false
} = {
  ignoreNotFound: true
}) {
  if (ids.length === 0) {
    return [];
  }
  const res = await soClient.bulkGet(ids.map(id => ({
    id,
    type: _constants.FLEET_PROXY_SAVED_OBJECT_TYPE
  })));
  return res.saved_objects.map(so => {
    if (so.error) {
      if (!ignoreNotFound || so.error.statusCode !== 404) {
        throw so.error;
      }
      return undefined;
    }
    return savedObjectToFleetProxy(so);
  }).filter(fleetProxyOrUndefined => typeof fleetProxyOrUndefined !== 'undefined');
}
async function updateRelatedSavedObject(soClient, esClient, fleetServerHosts, outputs, downloadSources) {
  await (0, _pMap.default)(fleetServerHosts, fleetServerHost => _fleet_server_host.fleetServerHostService.update(soClient, esClient, fleetServerHost.id, {
    ...(0, _lodash.omit)(fleetServerHost, 'id'),
    proxy_id: null
  }), {
    concurrency: _constants.MAX_CONCURRENT_FLEET_PROXIES_OPERATIONS
  });
  await (0, _pMap.default)(outputs, output => _output.outputService.update(soClient, esClient, output.id, {
    ...(0, _lodash.omit)(output, 'id'),
    proxy_id: null
  }), {
    concurrency: _constants.MAX_CONCURRENT_FLEET_PROXIES_OPERATIONS
  });
  await (0, _pMap.default)(downloadSources, downloadSource => _download_source.downloadSourceService.update(soClient, esClient, downloadSource.id, {
    ...(0, _lodash.omit)(downloadSource, 'id'),
    proxy_id: null
  }));
}
async function getFleetProxyRelatedSavedObjects(soClient, proxyId) {
  const [{
    items: fleetServerHosts
  }, {
    items: outputs
  }, {
    items: downloadSources
  }] = await Promise.all([_fleet_server_host.fleetServerHostService.listAllForProxyId(soClient, proxyId), _output.outputService.listAllForProxyId(soClient, proxyId), _download_source.downloadSourceService.listAllForProxyId(soClient, proxyId)]);
  return {
    fleetServerHosts,
    outputs,
    downloadSources
  };
}