"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sendAutomaticUpgradeAgentsActions = sendAutomaticUpgradeAgentsActions;
exports.sendUpgradeAgentAction = sendUpgradeAgentAction;
exports.sendUpgradeAgentsActions = sendUpgradeAgentsActions;
var _errors = require("../../errors");
var _constants = require("../../constants");
var _agent_namespaces = require("../spaces/agent_namespaces");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _actions = require("./actions");
var _crud = require("./crud");
var _upgrade_action_runner = require("./upgrade_action_runner");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function sendUpgradeAgentAction({
  soClient,
  esClient,
  agentId,
  version,
  sourceUri
}) {
  const now = new Date().toISOString();
  const data = {
    version,
    sourceURI: sourceUri
  };
  const agentPolicy = await (0, _crud.getAgentPolicyForAgent)(soClient, esClient, agentId);
  if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_managed) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot upgrade agent ${agentId} in hosted agent policy ${agentPolicy.id}`);
  }
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  await (0, _actions.createAgentAction)(esClient, soClient, {
    agents: [agentId],
    created_at: now,
    data,
    ack_data: data,
    type: 'UPGRADE',
    namespaces: [currentSpaceId]
  });
  await (0, _crud.updateAgent)(esClient, agentId, {
    upgraded_at: null,
    upgrade_started_at: now
  });
}
async function sendUpgradeAgentsActions(soClient, esClient, options) {
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  // Full set of agents
  const outgoingErrors = {};
  let givenAgents = [];
  if ('agents' in options) {
    givenAgents = options.agents;
  } else if ('agentIds' in options) {
    const maybeAgents = await (0, _crud.getAgentsById)(esClient, soClient, options.agentIds);
    for (const maybeAgent of maybeAgents) {
      if ('notFound' in maybeAgent) {
        outgoingErrors[maybeAgent.id] = new _errors.AgentReassignmentError(`Cannot find agent ${maybeAgent.id}`);
      } else {
        givenAgents.push(maybeAgent);
      }
    }
  } else if ('kuery' in options) {
    var _options$batchSize, _options$showInactive;
    const batchSize = (_options$batchSize = options.batchSize) !== null && _options$batchSize !== void 0 ? _options$batchSize : _constants.SO_SEARCH_LIMIT;
    const namespaceFilter = await (0, _agent_namespaces.agentsKueryNamespaceFilter)(currentSpaceId);
    const kuery = namespaceFilter ? `${namespaceFilter} AND ${options.kuery}` : options.kuery;
    const res = await (0, _crud.getAgentsByKuery)(esClient, soClient, {
      kuery,
      showAgentless: options.showAgentless,
      showInactive: (_options$showInactive = options.showInactive) !== null && _options$showInactive !== void 0 ? _options$showInactive : false,
      page: 1,
      perPage: batchSize
    });
    if (res.total <= batchSize) {
      givenAgents = res.agents;
    } else {
      return await new _upgrade_action_runner.UpgradeActionRunner(esClient, soClient, {
        ...options,
        batchSize,
        total: res.total,
        spaceId: currentSpaceId
      }, {
        pitId: await (0, _crud.openPointInTime)(esClient)
      }).runActionAsyncTask();
    }
  }
  return await (0, _upgrade_action_runner.upgradeBatch)(esClient, givenAgents, outgoingErrors, options, [currentSpaceId]);
}
async function sendAutomaticUpgradeAgentsActions(soClient, esClient, options) {
  var _options$force, _options$spaceIds;
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  return await (0, _upgrade_action_runner.upgradeBatch)(esClient, options.agents, {}, {
    ...options,
    isAutomatic: true,
    force: (_options$force = options.force) !== null && _options$force !== void 0 ? _options$force : false
  }, (_options$spaceIds = options.spaceIds) !== null && _options$spaceIds !== void 0 ? _options$spaceIds : [currentSpaceId]);
}