"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.putFleetServerHostHandler = exports.postFleetServerHost = exports.getFleetServerHostHandler = exports.getAllFleetServerHostsHandler = exports.deleteFleetServerHostHandler = void 0;
var _server = require("@kbn/core/server");
var _lodash = require("lodash");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _constants = require("../../constants");
var _errors = require("../../errors");
var _services = require("../../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function ensureNoDuplicateSecrets(fleetServerHost) {
  var _fleetServerHost$ssl, _fleetServerHost$secr, _fleetServerHost$secr2, _fleetServerHost$ssl2, _fleetServerHost$secr3, _fleetServerHost$secr4;
  if ((_fleetServerHost$ssl = fleetServerHost.ssl) !== null && _fleetServerHost$ssl !== void 0 && _fleetServerHost$ssl.key && (_fleetServerHost$secr = fleetServerHost.secrets) !== null && _fleetServerHost$secr !== void 0 && (_fleetServerHost$secr2 = _fleetServerHost$secr.ssl) !== null && _fleetServerHost$secr2 !== void 0 && _fleetServerHost$secr2.key) {
    throw _boom.default.badRequest('Cannot specify both ssl.key and secrets.ssl.key');
  }
  if ((_fleetServerHost$ssl2 = fleetServerHost.ssl) !== null && _fleetServerHost$ssl2 !== void 0 && _fleetServerHost$ssl2.es_key && (_fleetServerHost$secr3 = fleetServerHost.secrets) !== null && _fleetServerHost$secr3 !== void 0 && (_fleetServerHost$secr4 = _fleetServerHost$secr3.ssl) !== null && _fleetServerHost$secr4 !== void 0 && _fleetServerHost$secr4.es_key) {
    throw _boom.default.badRequest('Cannot specify both ssl.es_key and secrets.ssl.es_key');
  }
}
async function checkFleetServerHostsWriteAPIsAllowed(soClient, hostUrls) {
  const cloudSetup = _services.appContextService.getCloud();
  if (!(cloudSetup !== null && cloudSetup !== void 0 && cloudSetup.isServerlessEnabled)) {
    return;
  }

  // Fleet Server hosts must have the default host URL in serverless.
  const serverlessDefaultFleetServerHost = await _services.fleetServerHostService.get(soClient, _constants.SERVERLESS_DEFAULT_FLEET_SERVER_HOST_ID);
  if (!(0, _lodash.isEqual)(hostUrls, serverlessDefaultFleetServerHost.host_urls)) {
    throw new _errors.FleetServerHostUnauthorizedError(`Fleet server host must have default URL in serverless: ${serverlessDefaultFleetServerHost.host_urls}`);
  }
}
const postFleetServerHost = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;

  // In serverless, allow create fleet server host if host url is same as default.
  await checkFleetServerHostsWriteAPIsAllowed(soClient, request.body.host_urls);
  const {
    id,
    ...data
  } = request.body;
  ensureNoDuplicateSecrets(data);
  const FleetServerHost = await _services.fleetServerHostService.create(soClient, esClient, {
    ...data,
    is_preconfigured: false
  }, {
    id
  });
  if (FleetServerHost.is_default) {
    await _services.agentPolicyService.bumpAllAgentPolicies(esClient);
  }
  const body = {
    item: FleetServerHost
  };
  return response.ok({
    body
  });
};
exports.postFleetServerHost = postFleetServerHost;
const getFleetServerHostHandler = async (context, request, response) => {
  const soClient = (await context.core).savedObjects.client;
  try {
    const item = await _services.fleetServerHostService.get(soClient, request.params.itemId);
    const body = {
      item
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
      return response.notFound({
        body: {
          message: `Fleet server ${request.params.itemId} not found`
        }
      });
    }
    throw error;
  }
};
exports.getFleetServerHostHandler = getFleetServerHostHandler;
const deleteFleetServerHostHandler = async (context, request, response) => {
  try {
    const coreContext = await context.core;
    const soClient = coreContext.savedObjects.client;
    const esClient = coreContext.elasticsearch.client.asInternalUser;
    await _services.fleetServerHostService.delete(soClient, esClient, request.params.itemId);
    const body = {
      id: request.params.itemId
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
      return response.notFound({
        body: {
          message: `Fleet server ${request.params.itemId} not found`
        }
      });
    }
    throw error;
  }
};
exports.deleteFleetServerHostHandler = deleteFleetServerHostHandler;
const putFleetServerHostHandler = async (context, request, response) => {
  try {
    const coreContext = await context.core;
    const esClient = coreContext.elasticsearch.client.asInternalUser;
    const soClient = coreContext.savedObjects.client;

    // In serverless, allow update fleet server host if host url is same as default.
    if (request.body.host_urls) {
      await checkFleetServerHostsWriteAPIsAllowed(soClient, request.body.host_urls);
    }
    ensureNoDuplicateSecrets(request.body);
    const item = await _services.fleetServerHostService.update(soClient, esClient, request.params.itemId, request.body);
    const body = {
      item
    };
    if (item.is_default) {
      await _services.agentPolicyService.bumpAllAgentPolicies(esClient);
    } else {
      await _services.agentPolicyService.bumpAllAgentPoliciesForFleetServerHosts(esClient, item.id);
    }
    return response.ok({
      body
    });
  } catch (error) {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
      return response.notFound({
        body: {
          message: `Fleet server ${request.params.itemId} not found`
        }
      });
    }
    throw error;
  }
};
exports.putFleetServerHostHandler = putFleetServerHostHandler;
const getAllFleetServerHostsHandler = async (context, request, response) => {
  const soClient = (await context.core).savedObjects.client;
  const res = await _services.fleetServerHostService.list(soClient);
  const body = {
    items: res.items,
    page: res.page,
    perPage: res.perPage,
    total: res.total
  };
  return response.ok({
    body
  });
};
exports.getAllFleetServerHostsHandler = getAllFleetServerHostsHandler;