"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postEnrollmentApiKeyHandler = exports.getOneEnrollmentApiKeyHandler = exports.getEnrollmentApiKeysHandler = exports.deleteEnrollmentApiKeyHandler = void 0;
var _server = require("@kbn/core/server");
var APIKeyService = _interopRequireWildcard(require("../../services/api_keys"));
var _agent_policy = require("../../services/agent_policy");
var _errors = require("../../errors");
var _get_current_namespace = require("../../services/spaces/get_current_namespace");
var _helpers = require("../../services/spaces/helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getEnrollmentApiKeysHandler = async (context, request, response) => {
  // Use kibana_system and depend on authz checks on HTTP layer to prevent abuse
  const esClient = (await context.core).elasticsearch.client.asInternalUser;
  const soClient = (await context.core).savedObjects.client;
  const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
  const {
    items,
    total,
    page,
    perPage
  } = await APIKeyService.listEnrollmentApiKeys(esClient, {
    page: request.query.page,
    perPage: request.query.perPage,
    kuery: request.query.kuery,
    spaceId: useSpaceAwareness ? (0, _get_current_namespace.getCurrentNamespace)(soClient) : undefined
  });
  const body = {
    list: items,
    // deprecated
    items,
    total,
    page,
    perPage
  };
  return response.ok({
    body
  });
};
exports.getEnrollmentApiKeysHandler = getEnrollmentApiKeysHandler;
const postEnrollmentApiKeyHandler = async (context, request, response) => {
  const {
    elasticsearch,
    savedObjects
  } = await context.core;
  const soClient = savedObjects.client;
  const esClient = elasticsearch.client.asInternalUser;
  // validate policy exists in the current space
  await _agent_policy.agentPolicyService.get(soClient, request.body.policy_id).catch(err => {
    if (_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      throw new _errors.AgentPolicyNotFoundError(`Agent policy "${request.body.policy_id}" not found`);
    }
    throw err;
  });
  const apiKey = await APIKeyService.generateEnrollmentAPIKey(soClient, esClient, {
    name: request.body.name,
    expiration: request.body.expiration,
    agentPolicyId: request.body.policy_id
  });
  const body = {
    item: apiKey,
    action: 'created'
  };
  return response.ok({
    body
  });
};
exports.postEnrollmentApiKeyHandler = postEnrollmentApiKeyHandler;
const deleteEnrollmentApiKeyHandler = async (context, request, response) => {
  try {
    const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
    const coreContext = await context.core;
    const esClient = coreContext.elasticsearch.client.asInternalUser;
    const currentNamespace = (0, _get_current_namespace.getCurrentNamespace)(coreContext.savedObjects.client);
    await APIKeyService.deleteEnrollmentApiKey(esClient, request.params.keyId, false, useSpaceAwareness ? currentNamespace : undefined);
    const body = {
      action: 'deleted'
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `EnrollmentAPIKey ${request.params.keyId} not found`
        }
      });
    }
    throw error;
  }
};
exports.deleteEnrollmentApiKeyHandler = deleteEnrollmentApiKeyHandler;
const getOneEnrollmentApiKeyHandler = async (context, request, response) => {
  // Use kibana_system and depend on authz checks on HTTP layer to prevent abuse

  try {
    const coreContext = await context.core;
    const esClient = coreContext.elasticsearch.client.asInternalUser;
    const currentNamespace = (0, _get_current_namespace.getCurrentNamespace)(coreContext.savedObjects.client);
    const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
    const apiKey = await APIKeyService.getEnrollmentAPIKey(esClient, request.params.keyId, useSpaceAwareness ? currentNamespace : undefined);
    const body = {
      item: apiKey
    };
    return response.ok({
      body
    });
  } catch (error) {
    if (error.isBoom && error.output.statusCode === 404) {
      return response.notFound({
        body: {
          message: `EnrollmentAPIKey ${request.params.keyId} not found`
        }
      });
    }
    throw error;
  }
};
exports.getOneEnrollmentApiKeyHandler = getOneEnrollmentApiKeyHandler;