"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../common/constants");
var _api_privileges = require("../../constants/api_privileges");
var _constants2 = require("../../constants");
var _types = require("../../types");
var _experimental_features = require("../../../common/experimental_features");
var _errors = require("../schema/errors");
var _utils = require("../schema/utils");
var _handlers = require("./handlers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerRoutes = (router, config) => {
  // List - Fleet Server needs access to run setup
  router.versioned.get({
    path: _constants2.AGENT_POLICY_API_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Get agent policies`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetAgentPoliciesRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => (0, _utils.ListResponseSchema)(_types.AgentPolicyResponseSchema)
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getAgentPoliciesHandler);

  // Bulk GET
  router.versioned.post({
    path: _constants2.AGENT_POLICY_API_ROUTES.BULK_GET_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Bulk get agent policies`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.BulkGetAgentPoliciesRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.BulkGetAgentPoliciesResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.bulkGetAgentPoliciesHandler);

  // Get one
  router.versioned.get({
    path: _constants2.AGENT_POLICY_API_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Get an agent policy`,
    description: `Get an agent policy by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneAgentPolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getOneAgentPolicyHandler);
  const experimentalFeatures = (0, _experimental_features.parseExperimentalConfigValue)(config.enableExperimental);
  if (experimentalFeatures.enableAutomaticAgentUpgrades) {
    router.versioned.get({
      path: _constants2.AGENT_POLICY_API_ROUTES.AUTO_UPGRADE_AGENTS_STATUS_PATTERN,
      security: {
        authz: {
          requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
        }
      },
      summary: `Get auto upgrade agent status`,
      description: `Get auto upgrade agent status`,
      options: {
        tags: ['oas-tag:Elastic Agent policies']
      }
    }).addVersion({
      version: _constants.API_VERSIONS.public.v1,
      validate: {
        request: _types.GetAutoUpgradeAgentsStatusRequestSchema,
        response: {
          200: {
            description: 'OK: A successful request.',
            body: () => _types.GetAutoUpgradeAgentsStatusResponseSchema
          },
          400: {
            description: 'A bad request.',
            body: _errors.genericErrorResponse
          }
        }
      }
    }, _handlers.getAutoUpgradeAgentsStatusHandler);
  }

  // Create
  router.versioned.post({
    path: _constants2.AGENT_POLICY_API_ROUTES.CREATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
      }
    },
    summary: `Create an agent policy`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.CreateAgentPolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.createAgentPolicyHandler);

  // Create agent + package policies in a single request
  // Used for agentless integrations
  router.versioned.post({
    path: _constants2.AGENT_POLICY_API_ROUTES.CREATE_WITH_PACKAGE_POLICIES,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
      }
    },
    summary: `Create an agent policy and its package policies in one request`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.CreateAgentAndPackagePolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.createAgentAndPackagePoliciesHandler);

  // Update
  router.versioned.put({
    path: _constants2.AGENT_POLICY_API_ROUTES.UPDATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
      }
    },
    summary: `Update an agent policy`,
    description: `Update an agent policy by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.UpdateAgentPolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.updateAgentPolicyHandler);

  // Copy
  router.versioned.post({
    path: _constants2.AGENT_POLICY_API_ROUTES.COPY_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
      }
    },
    summary: `Copy an agent policy`,
    description: `Copy an agent policy by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.CopyAgentPolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.copyAgentPolicyHandler);

  // Delete
  router.versioned.post({
    path: _constants2.AGENT_POLICY_API_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL]
      }
    },
    summary: `Delete an agent policy`,
    description: `Delete an agent policy by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeleteAgentPolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.DeleteAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.deleteAgentPoliciesHandler);

  // Get one full agent policy
  router.versioned.get({
    path: _constants2.AGENT_POLICY_API_ROUTES.FULL_INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ]
      }
    },
    summary: `Get a full agent policy`,
    description: `Get a full agent policy by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetFullAgentPolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetFullAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getFullAgentPolicy);

  // Download one full agent policy
  router.versioned.get({
    path: _constants2.AGENT_POLICY_API_ROUTES.FULL_INFO_DOWNLOAD_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    enableQueryVersion: true,
    summary: `Download an agent policy`,
    description: `Download an agent policy by ID.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetFullAgentPolicyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.DownloadFullAgentPolicyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        },
        404: {
          description: 'Not found.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.downloadFullAgentPolicy);

  // Get agent manifest
  router.versioned.get({
    path: _constants.K8S_API_ROUTES.K8S_INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Get a full K8s agent manifest`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetK8sManifestRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetK8sManifestResponseScheme
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.getK8sManifest);

  // Download agent manifest
  router.versioned.get({
    path: _constants.K8S_API_ROUTES.K8S_DOWNLOAD_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    enableQueryVersion: true,
    summary: `Download an agent manifest`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetK8sManifestRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _configSchema.schema.string()
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        },
        404: {
          description: 'Not found.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.downloadK8sManifest);
  router.versioned.post({
    path: _constants2.AGENT_POLICY_API_ROUTES.LIST_OUTPUTS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ]
      }
    },
    summary: `Get outputs for agent policies`,
    description: `Get a list of outputs associated with agent policies.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetListAgentPolicyOutputsRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetListAgentPolicyOutputsResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.GetListAgentPolicyOutputsHandler);
  router.versioned.get({
    path: _constants2.AGENT_POLICY_API_ROUTES.INFO_OUTPUTS_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.READ, _api_privileges.FLEET_API_PRIVILEGES.SETTINGS.READ]
      }
    },
    summary: `Get outputs for an agent policy`,
    description: `Get a list of outputs associated with agent policy by policy id.`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetAgentPolicyOutputsRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.GetAgentPolicyOutputsResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.GetAgentPolicyOutputsHandler);
  router.versioned.post({
    path: _constants2.AGENT_POLICY_API_ROUTES.CLEANUP_REVISIONS_PATTERN,
    access: 'internal',
    enableQueryVersion: true,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENT_POLICIES.ALL, _api_privileges.FLEET_API_PRIVILEGES.AGENTS.READ]
      }
    },
    summary: `Run a task to cleanup excess agent policy revisions`,
    options: {
      tags: ['oas-tag:Elastic Agent policies']
    }
  }).addVersion({
    version: _constants.API_VERSIONS.internal.v1,
    validate: {
      request: _types.RunAgentPolicyRevisionsCleanupTaskRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.RunAgentPolicyRevisionsCleanupTaskResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handlers.RunAgentPolicyRevisionsCleanupTaskHandler);
};
exports.registerRoutes = registerRoutes;