"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.templateToPredicate = exports.replaceProperties = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Creates a predicate function which matches a single AST node against a
 * template object. The template object should have the same keys as the
 * AST node, and the values should be:
 *
 * - An array matches if the node key is in the array.
 * - A RegExp matches if the node key matches the RegExp.
 * - Any other value matches if the node key is triple-equal to the value.
 *
 * @param template Template from which to create a predicate function.
 * @returns A predicate function that matches nodes against the template.
 */
const templateToPredicate = template => {
  const keys = Object.keys(template);
  const predicate = node => {
    for (const key of keys) {
      const matcher = template[key];
      if (matcher instanceof Array) {
        if (!matcher.includes(node[key])) {
          return false;
        }
      } else if (matcher instanceof RegExp) {
        if (!matcher.test(String(node[key]))) {
          return false;
        }
      } else if (node[key] !== matcher) {
        return false;
      }
    }
    return true;
  };
  return predicate;
};
exports.templateToPredicate = templateToPredicate;
const replaceProperties = (obj, replacement) => {
  for (const key in obj) {
    if (typeof key === 'string' && Object.prototype.hasOwnProperty.call(obj, key)) delete obj[key];
  }
  Object.assign(obj, replacement);
};
exports.replaceProperties = replaceProperties;