"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNullCheckOperatorSuggestions = void 0;
exports.getOperatorSuggestion = getOperatorSuggestion;
exports.getOperatorsSuggestionsAfterNot = exports.getOperatorSuggestions = void 0;
exports.getSuggestionsToRightOfOperatorExpression = getSuggestionsToRightOfOperatorExpression;
exports.isArrayType = isArrayType;
var _complete_items = require("../../commands_registry/complete_items");
var _helpers = require("./autocomplete/helpers");
var _types = require("../types");
var _all_operators = require("../all_operators");
var _functions = require("./functions");
var _shared = require("./shared");
var _test_functions = require("./test_functions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getOperatorSuggestion(fn) {
  const hasArgs = fn.signatures.some(({
    params
  }) => params.length > 1);
  const suggestion = {
    label: fn.name.toUpperCase(),
    text: hasArgs ? `${fn.name.toUpperCase()} $0` : fn.name.toUpperCase(),
    asSnippet: hasArgs,
    kind: 'Operator',
    detail: fn.description,
    documentation: {
      value: ''
    },
    sortText: 'D'
  };
  return hasArgs ? (0, _helpers.withAutoSuggest)(suggestion) : suggestion;
}

/**
 * Builds suggestions for operators based on the provided predicates.
 *
 * @param predicates a set of conditions that must be met for an operator to be included in the suggestions
 * @returns
 */
const getOperatorSuggestions = (predicates, hasMinimumLicenseRequired, activeProduct) => {
  const filteredDefinitions = (0, _functions.filterFunctionDefinitions)((0, _test_functions.getTestFunctions)().length ? [..._all_operators.operatorsDefinitions, ...(0, _test_functions.getTestFunctions)()] : _all_operators.operatorsDefinitions, predicates, hasMinimumLicenseRequired, activeProduct);

  // make sure the operator has at least one signature that matches
  // the type of the existing left argument if provided (e.g. "doubleField <suggest>")
  return (predicates !== null && predicates !== void 0 && predicates.leftParamType ? filteredDefinitions.filter(({
    signatures
  }) => signatures.some(({
    params
  }) => !params.length || params.some(pArg => pArg.type === (predicates === null || predicates === void 0 ? void 0 : predicates.leftParamType) || pArg.type === 'any'))) : filteredDefinitions).map(getOperatorSuggestion);
};
exports.getOperatorSuggestions = getOperatorSuggestions;
const getOperatorsSuggestionsAfterNot = () => {
  return _all_operators.operatorsDefinitions.filter(({
    name
  }) => name === 'like' || name === 'rlike' || name === 'in').map(getOperatorSuggestion);
};
exports.getOperatorsSuggestionsAfterNot = getOperatorsSuggestionsAfterNot;
const getNullCheckOperators = () => {
  return _all_operators.operatorsDefinitions.filter(({
    name
  }) => name === 'is null' || name === 'is not null');
};

/** Suggest complete "IS [NOT] NULL" operators when the user has started typing "IS ..." */
const getNullCheckOperatorSuggestions = (queryText, location, leftParamType) => {
  const candidates = getNullCheckOperators().map(({
    name
  }) => name);
  const queryLower = queryText.toLowerCase();
  const queryNormalized = queryLower.replace(/\s+/g, ' ').replace(/\s+$/, ' ');
  const allowedOperators = candidates.filter(candidate => {
    const candidateLower = candidate.toLowerCase();
    // - "... is " matches candidate "is null" (prefix: "is ")
    // - "... is n" matches candidate "is not null" (prefix: "is n")
    return [...candidateLower].some((_, i) => queryNormalized.endsWith(candidateLower.slice(0, i + 1)));
  });
  return getOperatorSuggestions({
    location,
    leftParamType,
    allowed: allowedOperators
  });
};
exports.getNullCheckOperatorSuggestions = getNullCheckOperatorSuggestions;
function isArrayType(type) {
  return type.endsWith('[]');
}
function getSupportedTypesForBinaryOperators(fnDef, previousType) {
  // Retrieve list of all 'right' supported types that match the left hand side of the function
  return fnDef && Array.isArray(fnDef === null || fnDef === void 0 ? void 0 : fnDef.signatures) ? fnDef.signatures.filter(({
    params
  }) => params.find(p => p.name === 'left' && p.type === previousType)).map(({
    params
  }) => {
    var _params$;
    return (_params$ = params[1]) === null || _params$ === void 0 ? void 0 : _params$.type;
  }) : [previousType];
}

/**
 * This function is used to
 * - suggest the next argument for an incomplete or incorrect binary operator expression (e.g. field > <suggest>)
 * - suggest an operator to the right of a complete binary operator expression (e.g. field > 0 <suggest>)
 * - suggest an operator to the right of a complete unary operator (e.g. field IS NOT NULL <suggest>)
 *
 * TODO — is this function doing too much?
 */
async function getSuggestionsToRightOfOperatorExpression({
  queryText,
  location,
  rootOperator: operator,
  preferredExpressionType,
  getExpressionType,
  getColumnsByType,
  hasMinimumLicenseRequired,
  activeProduct
}) {
  const suggestions = [];
  const isFnComplete = (0, _functions.checkFunctionInvocationComplete)(operator, getExpressionType);
  if (isFnComplete.complete) {
    // i.e. ... | <COMMAND> field > 0 <suggest>
    // i.e. ... | <COMMAND> field + otherN <suggest>
    const operatorReturnType = getExpressionType(operator);
    suggestions.push(...getOperatorSuggestions({
      location,
      // here we use the operator return type because we're suggesting operators that could
      // accept the result of the existing operator as a left operand
      leftParamType: operatorReturnType === 'unknown' || operatorReturnType === 'unsupported' ? 'any' : operatorReturnType,
      ignored: ['=', ':'],
      allowed: operatorReturnType === 'boolean' ? [..._all_operators.logicalOperators.filter(({
        locationsAvailable
      }) => locationsAvailable.includes(location)).map(({
        name
      }) => name)] : undefined
    }));
  } else {
    // i.e. ... | <COMMAND> field >= <suggest>
    // i.e. ... | <COMMAND> field + <suggest>
    // i.e. ... | <COMMAND> field and <suggest>

    // Because it's an incomplete function, need to extract the type of the current argument
    // and suggest the next argument based on types

    // pick the last arg and check its type to verify whether is incomplete for the given function
    const cleanedArgs = (0, _shared.removeFinalUnknownIdentiferArg)(operator.args, getExpressionType);
    const leftArgType = getExpressionType(operator.args[cleanedArgs.length - 1]);
    if (isFnComplete.reason === 'tooFewArgs') {
      const fnDef = (0, _functions.getFunctionDefinition)(operator.name);
      if (fnDef !== null && fnDef !== void 0 && fnDef.signatures.every(({
        params
      }) => params.some(({
        type
      }) => isArrayType(type)))) {
        suggestions.push(_complete_items.listCompleteItem);
      } else {
        var _getFunctionDefinitio;
        const finalType = leftArgType || 'any';
        const supportedTypes = getSupportedTypesForBinaryOperators(fnDef, finalType);

        // this is a special case with AND/OR
        // <COMMAND> expression AND/OR <suggest>
        // technically another boolean value should be suggested, but it is a better experience
        // to actually suggest a wider set of fields/functions
        const typeToUse = finalType === 'boolean' && ((_getFunctionDefinitio = (0, _functions.getFunctionDefinition)(operator.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === _types.FunctionDefinitionTypes.OPERATOR ? ['any'] : supportedTypes;
        const couldBeNullCheck = getNullCheckOperators().some(({
          name
        }) => name.startsWith(operator.name.toLowerCase()));
        if (couldBeNullCheck) {
          suggestions.push(...getNullCheckOperatorSuggestions(queryText, location, finalType === 'unknown' || finalType === 'unsupported' ? 'any' : finalType));
        } else {
          // TODO replace with fields callback + function suggestions
          suggestions.push(...(await (0, _helpers.getFieldsOrFunctionsSuggestions)(typeToUse, location, getColumnsByType, {
            functions: true,
            columns: true,
            values: Boolean(operator.subtype === 'binary-expression')
          }, {}, hasMinimumLicenseRequired, activeProduct)));
        }
      }
    }

    /**
     * If the caller has supplied a preferred expression type, we can suggest operators that
     * would move the user toward that expression type.
     *
     * e.g. if we have a preferred type of boolean and we have `timestamp > "2002" AND doubleField`
     * this is an incorrect signature for AND because the left side is boolean and the right side is double
     *
     * Knowing that we prefer boolean expressions, we suggest operators that would accept doubleField as a left operand
     * and also return a boolean value.
     *
     * I believe this is only used in WHERE and probably bears some rethinking.
     */
    if (isFnComplete.reason === 'wrongTypes') {
      if (leftArgType && preferredExpressionType) {
        // suggest something to complete the operator
        if (leftArgType !== preferredExpressionType && (0, _types.isParameterType)(leftArgType) && (0, _types.isReturnType)(preferredExpressionType)) {
          suggestions.push(...getOperatorSuggestions({
            location,
            leftParamType: leftArgType,
            returnTypes: [preferredExpressionType]
          }));
        }
      }
    }
  }
  return suggestions.map(s => {
    const overlap = (0, _shared.getOverlapRange)(queryText, s.text);
    return {
      ...s,
      rangeToReplace: overlap
    };
  });
}