"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CommandRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Interface defining the methods that each ES|QL command should register.
 * These methods provide functionality specific to the command's behavior.
 *
 * @template TContext The type of any additional context required by the methods.
 */

/**
 * Interface defining the structure of a single registered command.
 */

/**
 * Interface for the ESQL Command Registry, defining its public API.
 */

/**
 * Implementation of the ESQL Command Registry.
 * This class manages the registration, storage, and retrieval of ESQL command methods.
 */
class CommandRegistry {
  constructor() {
    (0, _defineProperty2.default)(this, "commands", new Map());
    this.commands = new Map();
  }

  /**
   * Registers a new command and its methods into the registry.
   * @param command The command object containing its name and methods.
   */
  registerCommand(command) {
    if (!this.commands.has(command.name)) {
      this.commands.set(command.name, {
        methods: command.methods,
        metadata: command.metadata
      });
    }
  }

  /**
   * Retrieves the methods associated with a specific command name.
   * @param commandName The name of the command to look up.
   * @returns The ICommandMethods object if the command is found, otherwise undefined.
   */
  getCommandMethods(commandName) {
    var _this$commands$get;
    return (_this$commands$get = this.commands.get(commandName)) === null || _this$commands$get === void 0 ? void 0 : _this$commands$get.methods;
  }

  /**
   * Returns a list of all command names currently registered in the registry.
   * @returns An array of command names.
   */
  getAllCommandNames() {
    return Array.from(this.commands.keys());
  }

  /**
   * Retrieves all registered commands, including their methods and metadata.
   * @returns An array of ICommand objects representing all registered commands.
   */
  getAllCommands() {
    return Array.from(this.commands.entries()).map(([name, {
      methods,
      metadata
    }]) => ({
      name,
      methods,
      metadata
    }));
  }

  /**
   * Retrieves a command by its name, including its methods and optional metadata.
   * @param commandName The name of the command to retrieve.
   * @returns The ICommand object if found, otherwise undefined.
   */
  getCommandByName(commandName) {
    const command = this.commands.get(commandName);
    return command ? {
      name: commandName,
      methods: command.methods,
      metadata: command.metadata
    } : undefined;
  }
}
exports.CommandRegistry = CommandRegistry;