"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAndInstallTemplates = createAndInstallTemplates;
exports.deleteTemplate = deleteTemplate;
exports.deleteTemplates = deleteTemplates;
exports.installEntityManagerTemplates = void 0;
exports.removeIlmInServerless = removeIlmInServerless;
exports.upsertComponent = upsertComponent;
exports.upsertTemplate = upsertTemplate;
var _base_latest = require("../templates/components/base_latest");
var _base_history = require("../templates/components/base_history");
var _entity = require("../templates/components/entity");
var _event = require("../templates/components/event");
var _retry = require("./entities/helpers/retry");
var _entities_latest_template = require("./entities/templates/entities_latest_template");
var _entities_history_template = require("./entities/templates/entities_history_template");
var _entities_reset_template = require("./entities/templates/entities_reset_template");
var _entities_updates_template = require("./entities/templates/entities_updates_template");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const installEntityManagerTemplates = async ({
  esClient,
  logger,
  isServerless
}) => {
  await Promise.all([upsertComponent({
    esClient,
    logger,
    component: removeIlmInServerless(_base_latest.entitiesLatestBaseComponentTemplateConfig, isServerless)
  }), upsertComponent({
    esClient,
    logger,
    component: removeIlmInServerless(_base_history.entitiesHistoryBaseComponentTemplateConfig, isServerless)
  }), upsertComponent({
    esClient,
    logger,
    component: removeIlmInServerless(_event.entitiesEventComponentTemplateConfig, isServerless)
  }), upsertComponent({
    esClient,
    logger,
    component: removeIlmInServerless(_entity.entitiesEntityComponentTemplateConfig, isServerless)
  })]);
};
exports.installEntityManagerTemplates = installEntityManagerTemplates;
async function upsertTemplate({
  esClient,
  template,
  logger
}) {
  try {
    const result = await (0, _retry.retryTransientEsErrors)(() => esClient.indices.putIndexTemplate(template), {
      logger
    });
    logger.debug(() => `Installed entity manager index template: ${JSON.stringify(template)}`);
    return result;
  } catch (error) {
    logger.error(`Error updating entity manager index template: ${error.message}`);
    throw error;
  }
}
async function createAndInstallTemplates(esClient, definition, logger) {
  const latestTemplate = (0, _entities_latest_template.generateEntitiesLatestIndexTemplateConfig)(definition);
  const historyTemplate = (0, _entities_history_template.generateEntitiesHistoryIndexTemplateConfig)(definition);
  const resetTemplate = (0, _entities_reset_template.generateEntitiesResetIndexTemplateConfig)(definition);
  const updatesTemplate = (0, _entities_updates_template.generateEntitiesUpdatesIndexTemplateConfig)(definition);
  await Promise.all([upsertTemplate({
    esClient,
    template: latestTemplate,
    logger
  }), upsertTemplate({
    esClient,
    template: historyTemplate,
    logger
  }), upsertTemplate({
    esClient,
    template: resetTemplate,
    logger
  }), upsertTemplate({
    esClient,
    template: updatesTemplate,
    logger
  })]);
  return [{
    type: 'template',
    id: latestTemplate.name
  }, {
    type: 'template',
    id: historyTemplate.name
  }, {
    type: 'template',
    id: resetTemplate.name
  }, {
    type: 'template',
    id: updatesTemplate.name
  }];
}
async function deleteTemplate({
  esClient,
  name,
  logger
}) {
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.indices.deleteIndexTemplate({
      name
    }, {
      ignore: [404]
    }), {
      logger
    });
  } catch (error) {
    logger.error(`Error deleting entity manager index template: ${error.message}`);
    throw error;
  }
}
async function deleteTemplates(esClient, definition, logger) {
  try {
    var _definition$installed;
    await Promise.all(((_definition$installed = definition.installedComponents) !== null && _definition$installed !== void 0 ? _definition$installed : []).filter(({
      type
    }) => type === 'template').map(({
      id
    }) => (0, _retry.retryTransientEsErrors)(() => esClient.indices.deleteIndexTemplate({
      name: id
    }, {
      ignore: [404]
    }), {
      logger
    })));
  } catch (error) {
    logger.error(`Error deleting entity manager index template: ${error.message}`);
    throw error;
  }
}
async function upsertComponent({
  esClient,
  component,
  logger
}) {
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.cluster.putComponentTemplate(component), {
      logger
    });
    logger.debug(() => `Installed entity manager component template: ${JSON.stringify(component)}`);
  } catch (error) {
    logger.error(`Error updating entity manager component template: ${error.message}`);
    throw error;
  }
}
function removeIlmInServerless(component, isServerless) {
  if (isServerless) {
    var _component$template$s;
    component.template.lifecycle = undefined;
    if (((_component$template$s = component.template.settings) === null || _component$template$s === void 0 ? void 0 : _component$template$s.index) !== undefined) {
      component.template.settings.index.lifecycle = undefined;
    }
  }
  return component;
}