"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findAttackDiscoverySchedulesRoute = void 0;
var _actions = require("@kbn/security-solution-features/actions");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _build_response = require("../../../../../lib/build_response");
var _helpers = require("../../../../helpers");
var _throw_if_public_api_disabled = require("../../../helpers/throw_if_public_api_disabled");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_PAGE = 1;
const DEFAULT_PER_PAGE = 10;
const findAttackDiscoverySchedulesRoute = router => {
  router.versioned.get({
    access: 'public',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_SCHEDULES_FIND,
    security: {
      authz: {
        requiredPrivileges: [_actions.ATTACK_DISCOVERY_API_ACTION_ALL]
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.public.v1,
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.FindAttackDiscoverySchedulesRequestQuery)
      },
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.FindAttackDiscoverySchedulesResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }
    try {
      var _request$query$page, _request$query$per_pa;
      await (0, _throw_if_public_api_disabled.throwIfPublicApiDisabled)(context);
      const dataClient = await assistantContext.getAttackDiscoverySchedulingDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const results = await dataClient.findSchedules({
        page: request.query.page,
        perPage: request.query.per_page,
        sort: {
          sortField: request.query.sort_field,
          sortDirection: request.query.sort_direction
        }
      });

      // Transform internal format results to API format
      const apiResults = {
        page: (_request$query$page = request.query.page) !== null && _request$query$page !== void 0 ? _request$query$page : DEFAULT_PAGE,
        per_page: (_request$query$per_pa = request.query.per_page) !== null && _request$query$per_pa !== void 0 ? _request$query$per_pa : DEFAULT_PER_PAGE,
        total: results.total,
        data: results.data.map(schedule => (0, _elasticAssistantCommon.transformAttackDiscoveryScheduleToApi)(schedule))
      };
      return response.ok({
        body: apiResults
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.findAttackDiscoverySchedulesRoute = findAttackDiscoverySchedulesRoute;