"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.evaluateDefendInsights = void 0;
var _std = require("@kbn/std");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _server = require("@kbn/langchain/server");
var _utils = require("../../../routes/evaluate/utils");
var _utils2 = require("../../../routes/utils");
var _constants = require("../../attack_discovery/evaluation/constants");
var _prompts = require("../graphs/default_defend_insights_graph/prompts");
var _run_evaluations = require("./run_evaluations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const evaluateDefendInsights = async ({
  actionsClient,
  defendInsightsGraphs,
  anonymizationFields = _constants.DEFAULT_EVAL_ANONYMIZATION_FIELDS,
  // determines which fields are included in the alerts
  connectors,
  connectorTimeout,
  datasetName,
  esClient,
  soClient,
  kbDataClient,
  esClientInternalUser,
  evaluationId,
  evaluatorConnectorId,
  langSmithApiKey,
  langSmithProject,
  logger,
  runName,
  size
}) => {
  await (0, _std.asyncForEach)(defendInsightsGraphs, async ({
    getDefaultDefendInsightsGraph,
    insightType
  }) => {
    // create a graph for every connector:
    const graphs = await Promise.all(connectors.map(async connector => {
      var _connector$config;
      const llmType = (0, _utils2.getLlmType)(connector.actionTypeId);
      const prompts = await (0, _prompts.getDefendInsightsPrompt)({
        type: insightType,
        actionsClient,
        connectorId: connector.id,
        connector,
        savedObjectsClient: soClient
      });
      const traceOptions = {
        projectName: langSmithProject,
        tracers: [...(0, _langsmith.getLangSmithTracer)({
          apiKey: langSmithApiKey,
          projectName: langSmithProject,
          logger
        })]
      };
      const llm = new _server.ActionsClientLlm({
        actionsClient,
        connectorId: connector.id,
        llmType,
        logger,
        temperature: 0,
        // zero temperature for defend insights, because we want structured JSON output
        timeout: connectorTimeout,
        traceOptions,
        model: (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.defaultModel
      });
      const graph = getDefaultDefendInsightsGraph({
        insightType,
        endpointIds: [],
        // Empty endpointIds, because we are seeding the graph with the dataset
        esClient,
        kbDataClient: kbDataClient || null,
        llm,
        logger,
        size,
        anonymizationFields,
        prompts
      });
      return {
        connector,
        graph,
        llmType,
        name: `${runName} - ${connector.name} - ${evaluationId} - Defend Insights`,
        traceOptions
      };
    }));

    // run the evaluations for each graph:
    await (0, _run_evaluations.runDefendInsightsEvaluations)({
      insightType,
      evaluatorConnectorId,
      datasetName,
      graphs,
      langSmithApiKey,
      logger
    });
  });
  await (0, _utils.createOrUpdateEvaluationResults)({
    evaluationResults: [{
      id: evaluationId,
      status: _utils.EvaluationStatus.COMPLETE
    }],
    esClientInternalUser,
    logger
  });
};
exports.evaluateDefendInsights = evaluateDefendInsights;