"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateDelay = void 0;
exports.createResourceInstallationHelper = createResourceInstallationHelper;
exports.successResult = exports.getShouldRetry = exports.errorResult = void 0;
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// get multiples of 2 min
const DEFAULT_RETRY_BACKOFF_PER_ATTEMPT = 2 * 60 * 1000;
/**
 * Helper function that queues up resources to initialize until we are
 * ready to begin initialization. Once we're ready, we start taking from
 * the queue and kicking off initialization.
 *
 * If a resource is added after we begin initialization, we push it onto
 * the queue and the running loop will handle it
 *
 * If a resource is added to the queue when the processing loop is not
 * running, kick off the processing loop
 */
function createResourceInstallationHelper(logger, commonResourcesInitPromise, installFn) {
  let commonInitPromise = commonResourcesInitPromise;
  const initializedResources = new Map();
  const lastRetry = new Map();
  const waitUntilResourcesInstalled = async (namespace = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING, timeoutMs) => {
    try {
      const {
        result: commonInitResult,
        error: commonInitError
      } = await commonInitPromise;
      if (commonInitResult) {
        await installFn(namespace, timeoutMs);
        return successResult();
      } else {
        logger.warn(`Common resources were not initialized, cannot initialize resources for ${namespace}`);
        return errorResult(commonInitError);
      }
    } catch (err) {
      logger.warn(`Error initializing resources ${namespace} - ${err.message}`);
      return errorResult(err.message);
    }
  };
  return {
    add: (namespace = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING, timeoutMs) => {
      initializedResources.set(`${namespace}`,
      // Return a promise than can be checked when needed
      waitUntilResourcesInstalled(namespace, timeoutMs));
    },
    retry: (namespace = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING, initPromise, timeoutMs) => {
      const key = namespace;
      // Use the new common initialization promise if specified
      if (initPromise) {
        commonInitPromise = initPromise;
      }

      // Check the last retry time to see if we want to throttle this attempt
      const retryInfo = lastRetry.get(key);
      const shouldRetry = retryInfo ? getShouldRetry(retryInfo) : true;
      if (shouldRetry) {
        var _retryInfo$attempts;
        logger.info(`Retrying resource initialization for "${namespace}"`);
        // Update the last retry information
        lastRetry.set(key, {
          time: new Date().toISOString(),
          attempts: ((_retryInfo$attempts = retryInfo === null || retryInfo === void 0 ? void 0 : retryInfo.attempts) !== null && _retryInfo$attempts !== void 0 ? _retryInfo$attempts : 0) + 1
        });
        initializedResources.set(key,
        // Return a promise than can be checked when needed
        waitUntilResourcesInstalled(namespace, timeoutMs));
      }
    },
    getInitializedResources: async namespace => {
      const key = namespace;
      return initializedResources.has(key) ? await initializedResources.get(key) : errorResult(`Unrecognized spaceId ${key}`);
    }
  };
}
const successResult = () => ({
  result: true
});
exports.successResult = successResult;
const errorResult = error => ({
  result: false,
  error
});
exports.errorResult = errorResult;
const getShouldRetry = ({
  time,
  attempts
}) => {
  const now = new Date().valueOf();
  const nextRetryDate = new Date(time).valueOf() + calculateDelay(attempts);
  return now > nextRetryDate;
};
exports.getShouldRetry = getShouldRetry;
const calculateDelay = attempts => {
  if (attempts === 1) {
    return 30 * 1000; // 30s
  } else {
    // 2, 4, 6, 8, etc minutes
    return DEFAULT_RETRY_BACKOFF_PER_ATTEMPT * Math.pow(2, attempts - 2);
  }
};
exports.calculateDelay = calculateDelay;