"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getStreamObservable = exports.getPlaceholderObservable = void 0;
var _rxjs = require("rxjs");
var _translations = require("../translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MIN_DELAY = 10;
/**
 * Returns an Observable that reads data from a ReadableStream and emits values representing the state of the data processing.
 *
 * @param isError - indicates whether the reader response is an error message or not
 * @param reader - The ReadableStreamDefaultReader used to read data from the stream.
 * @param setLoading - A function to update the loading state.
 * @returns {Observable<PromptObservableState>} An Observable that emits PromptObservableState
 */
const getStreamObservable = ({
  isError,
  reader,
  setLoading
}) => new _rxjs.Observable(observer => {
  observer.next({
    chunks: [],
    loading: true
  });
  const decoder = new TextDecoder();
  const chunks = [];
  // Initialize an empty string to store the LangChain buffer.
  let langChainBuffer = '';

  // read data from LangChain stream
  function readLangChain() {
    reader.read().then(({
      done,
      value
    }) => {
      try {
        if (done) {
          if (langChainBuffer) {
            const finalChunk = getLangChainChunks([langChainBuffer])[0];
            if (finalChunk && finalChunk.length > 0) chunks.push(finalChunk);
          }
          observer.next({
            chunks,
            message: chunks.join(''),
            loading: false
          });
          observer.complete();
          return;
        }
        const decoded = decoder.decode(value);
        let nextChunks;
        if (isError) {
          nextChunks = [`${_translations.API_ERROR}\n\n${JSON.parse(decoded).message}`];
          nextChunks.forEach(chunk => {
            chunks.push(chunk);
            observer.next({
              chunks,
              message: chunks.join(''),
              loading: true
            });
          });
        } else {
          const output = decoded;
          const lines = output.split('\n');
          lines[0] = langChainBuffer + lines[0];
          langChainBuffer = lines.pop() || '';
          nextChunks = getLangChainChunks(lines);
          nextChunks.forEach(chunk => {
            chunks.push(chunk);
            observer.next({
              chunks,
              message: chunks.join(''),
              loading: true
            });
          });
        }
      } catch (err) {
        observer.error(err);
        return;
      }
      readLangChain();
    }).catch(err => {
      observer.error(err);
    });
  }
  readLangChain();
  return () => {
    reader.cancel();
  };
}).pipe(
// append a timestamp of when each value was emitted
(0, _rxjs.timestamp)(),
// use the previous timestamp to calculate a target
// timestamp for emitting the next value
(0, _rxjs.scan)((acc, value) => {
  const lastTimestamp = acc.timestamp || 0;
  const emitAt = Math.max(lastTimestamp + MIN_DELAY, value.timestamp);
  return {
    timestamp: emitAt,
    value: value.value
  };
}),
// add the delay based on the elapsed time
// using concatMap(of(value).pipe(delay(50))
// leads to browser issues because timers
// are throttled when the tab is not active
(0, _rxjs.concatMap)(value => {
  const now = Date.now();
  const delayFor = value.timestamp - now;
  if (delayFor <= 0) {
    return (0, _rxjs.of)(value.value);
  }
  return (0, _rxjs.of)(value.value).pipe((0, _rxjs.delay)(delayFor));
}),
// set loading to false when the observable completes or errors out
(0, _rxjs.finalize)(() => setLoading(false)));

/**
 * Parses a LangChain response from a string.
 * @param lines
 * @returns {string[]} - Parsed string array from the LangChain response.
 */
exports.getStreamObservable = getStreamObservable;
const getLangChainChunks = lines => lines.reduce((acc, b) => {
  if (b.length) {
    try {
      const obj = JSON.parse(b);
      if (obj.type === 'content' && obj.payload.length > 0) {
        return [...acc, obj.payload];
      }
      return acc;
    } catch (e) {
      return acc;
    }
  }
  return acc;
}, []);
const getPlaceholderObservable = () => new _rxjs.Observable();
exports.getPlaceholderObservable = getPlaceholderObservable;