"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStateContainer = getDataStateContainer;
var _rxjs = require("rxjs");
var _common = require("@kbn/inspector-plugin/common");
var _esQuery = require("@kbn/es-query");
var _discoverUtils = require("@kbn/discover-utils");
var _timerange = require("@kbn/timerange");
var _get_esql_data_view = require("./utils/get_esql_data_view");
var _types = require("../../types");
var _validate_time_range = require("./utils/validate_time_range");
var _fetch_all = require("../data_fetching/fetch_all");
var _use_saved_search_messages = require("../hooks/use_saved_search_messages");
var _get_fetch_observable = require("../data_fetching/get_fetch_observable");
var _get_default_profile_state = require("./utils/get_default_profile_state");
var _redux = require("./redux");
var _build_esql_fetch_subscribe = require("./utils/build_esql_fetch_subscribe");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Container responsible for fetching of data in Discover Main
 * Either by triggering requests to Elasticsearch directly, or by
 * orchestrating unified plugins / components like the histogram
 */
function getDataStateContainer({
  services,
  searchSessionManager,
  appStateContainer,
  internalState,
  runtimeStateManager,
  savedSearchContainer,
  setDataView,
  injectCurrentTab,
  getCurrentTab
}) {
  const {
    data,
    uiSettings,
    toastNotifications
  } = services;
  const {
    timefilter
  } = data.query.timefilter;
  const inspectorAdapters = {
    requests: new _common.RequestAdapter()
  };
  const fetchChart$ = new _rxjs.ReplaySubject(1);
  const disableNextFetchOnStateChange$ = new _rxjs.BehaviorSubject(false);
  let numberOfFetches = 0;

  /**
   * The observable to trigger data fetching in UI
   * By refetch$.next('reset') rows and fieldcounts are reset to allow e.g. editing of runtime fields
   * to be processed correctly
   */
  const refetch$ = new _rxjs.Subject();
  const getInitialFetchStatus = () => {
    const shouldSearchOnPageLoad = uiSettings.get(_discoverUtils.SEARCH_ON_PAGE_LOAD_SETTING) || savedSearchContainer.getState().id !== undefined || !timefilter.getRefreshInterval().pause || searchSessionManager.hasSearchSessionIdInURL();
    return shouldSearchOnPageLoad ? _types.FetchStatus.LOADING : _types.FetchStatus.UNINITIALIZED;
  };

  /**
   * The observables the UI (aka React component) subscribes to get notified about
   * the changes in the data fetching process (high level: fetching started, data was received)
   */
  const initialState = {
    fetchStatus: getInitialFetchStatus()
  };
  const dataSubjects = {
    main$: new _rxjs.BehaviorSubject(initialState),
    documents$: new _rxjs.BehaviorSubject(initialState),
    totalHits$: new _rxjs.BehaviorSubject(initialState)
  };

  // This is debugging code, helping you to understand which messages are sent to the data observables
  // Adding a debugger in the functions can be helpful to understand what triggers a message
  // dataSubjects.main$.subscribe((msg) => addLog('dataSubjects.main$', msg));
  // dataSubjects.documents$.subscribe((msg) => addLog('dataSubjects.documents$', msg));
  // dataSubjects.totalHits$.subscribe((msg) => addLog('dataSubjects.totalHits$', msg););
  // Add window.ELASTIC_DISCOVER_LOGGER = 'debug' to see messages in console

  /**
   * Subscribes to ES|QL fetches to handle state changes when loading or before a fetch completes
   */
  const {
    esqlFetchSubscribe,
    cleanupEsql
  } = (0, _build_esql_fetch_subscribe.buildEsqlFetchSubscribe)({
    internalState,
    appStateContainer,
    dataSubjects,
    injectCurrentTab
  });

  // The main subscription to handle state changes
  dataSubjects.documents$.pipe((0, _rxjs.switchMap)(esqlFetchSubscribe)).subscribe();
  // Make sure to clean up the ES|QL state when the saved search changes
  savedSearchContainer.getInitial$().subscribe(cleanupEsql);

  /**
   * handler emitted by `timefilter.getAutoRefreshFetch$()`
   * to notify when data completed loading and to start a new autorefresh loop
   */
  let autoRefreshDone = null;
  const setAutoRefreshDone = fn => {
    autoRefreshDone = fn;
  };
  const fetch$ = (0, _get_fetch_observable.getFetch$)({
    setAutoRefreshDone,
    data,
    main$: dataSubjects.main$,
    refetch$,
    searchSource: savedSearchContainer.getState().searchSource,
    searchSessionManager
  }).pipe((0, _rxjs.filter)(() => (0, _validate_time_range.validateTimeRange)(timefilter.getTime(), toastNotifications)), (0, _rxjs.tap)(() => inspectorAdapters.requests.reset()), (0, _rxjs.map)(val => ({
    options: {
      reset: val === 'reset',
      fetchMore: val === 'fetch_more'
    }
  })), (0, _rxjs.share)());
  let abortController;
  let abortControllerFetchMore;
  function subscribe() {
    const subscription = fetch$.pipe((0, _rxjs.mergeMap)(async ({
      options
    }) => {
      var _abortController, _abortControllerFetch;
      numberOfFetches += 1;
      const {
        id: currentTabId,
        resetDefaultProfileState,
        dataRequestParams
      } = getCurrentTab();
      const {
        scopedProfilesManager$,
        scopedEbtManager$,
        currentDataView$
      } = (0, _redux.selectTabRuntimeState)(runtimeStateManager, currentTabId);
      const scopedProfilesManager = scopedProfilesManager$.getValue();
      const scopedEbtManager = scopedEbtManager$.getValue();
      let searchSessionId;
      let isSearchSessionRestored;
      if (options.fetchMore && dataRequestParams.searchSessionId) {
        searchSessionId = dataRequestParams.searchSessionId;
        isSearchSessionRestored = dataRequestParams.isSearchSessionRestored;
      } else {
        ({
          searchSessionId,
          isSearchSessionRestored
        } = searchSessionManager.getNextSearchSessionId());
      }
      const commonFetchParams = {
        dataSubjects,
        initialFetchStatus: getInitialFetchStatus(),
        inspectorAdapters,
        searchSessionId,
        services,
        appStateContainer,
        internalState,
        savedSearch: savedSearchContainer.getState(),
        scopedProfilesManager,
        scopedEbtManager
      };
      (_abortController = abortController) === null || _abortController === void 0 ? void 0 : _abortController.abort();
      (_abortControllerFetch = abortControllerFetchMore) === null || _abortControllerFetch === void 0 ? void 0 : _abortControllerFetch.abort();
      if (options.fetchMore) {
        abortControllerFetchMore = new AbortController();
        const fetchMoreTracker = scopedEbtManager.trackPerformanceEvent('discoverFetchMore');

        // Calculate query range in seconds
        const timeRange = timefilter.getAbsoluteTime();
        const queryRangeSeconds = (0, _timerange.getTimeDifferenceInSeconds)(timeRange);
        await (0, _fetch_all.fetchMoreDocuments)({
          ...commonFetchParams,
          abortController: abortControllerFetchMore
        });
        fetchMoreTracker.reportEvent({
          key1: 'query_range_secs',
          value1: queryRangeSeconds
        });
        return;
      }
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setDataRequestParams)({
        dataRequestParams: {
          timeRangeAbsolute: timefilter.getAbsoluteTime(),
          timeRangeRelative: timefilter.getTime(),
          searchSessionId,
          isSearchSessionRestored
        }
      }));
      await scopedProfilesManager.resolveDataSourceProfile({
        dataSource: appStateContainer.getState().dataSource,
        dataView: savedSearchContainer.getState().searchSource.getField('index'),
        query: appStateContainer.getState().query
      });
      const dataView = currentDataView$.getValue();
      const defaultProfileState = dataView ? (0, _get_default_profile_state.getDefaultProfileState)({
        scopedProfilesManager,
        resetDefaultProfileState,
        dataView
      }) : undefined;
      const preFetchStateUpdate = defaultProfileState === null || defaultProfileState === void 0 ? void 0 : defaultProfileState.getPreFetchState();
      if (preFetchStateUpdate) {
        disableNextFetchOnStateChange$.next(true);
        await appStateContainer.replaceUrlState(preFetchStateUpdate);
        disableNextFetchOnStateChange$.next(false);
      }
      abortController = new AbortController();

      // Trigger chart fetching after the pre fetch state has been updated
      // to ensure state values that would affect data fetching are set
      fetchChart$.next();
      const prevAutoRefreshDone = autoRefreshDone;
      const fetchAllTracker = scopedEbtManager.trackPerformanceEvent('discoverFetchAll');

      // Calculate query range in seconds
      const timeRange = timefilter.getAbsoluteTime();
      const queryRangeSeconds = (0, _timerange.getTimeDifferenceInSeconds)(timeRange);
      await (0, _fetch_all.fetchAll)({
        ...commonFetchParams,
        reset: options.reset,
        abortController,
        getCurrentTab,
        onFetchRecordsComplete: async () => {
          const {
            resetDefaultProfileState: currentResetDefaultProfileState
          } = getCurrentTab();
          if (currentResetDefaultProfileState.resetId !== resetDefaultProfileState.resetId) {
            return;
          }
          const {
            esqlQueryColumns
          } = dataSubjects.documents$.getValue();
          const defaultColumns = uiSettings.get(_discoverUtils.DEFAULT_COLUMNS_SETTING, []);
          const postFetchStateUpdate = defaultProfileState === null || defaultProfileState === void 0 ? void 0 : defaultProfileState.getPostFetchState({
            defaultColumns,
            esqlQueryColumns
          });
          if (postFetchStateUpdate) {
            await appStateContainer.replaceUrlState(postFetchStateUpdate);
          }

          // Clear the default profile state flags after the data fetching
          // is done so refetches don't reset the state again
          internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setResetDefaultProfileState)({
            resetDefaultProfileState: {
              columns: false,
              rowHeight: false,
              breakdownField: false,
              hideChart: false
            }
          }));
        }
      });
      fetchAllTracker.reportEvent({
        key1: 'query_range_secs',
        value1: queryRangeSeconds
      });

      // If the autoRefreshCallback is still the same as when we started i.e. there was no newer call
      // replacing this current one, call it to make sure we tell that the auto refresh is done
      // and a new one can be scheduled. null is checked to always start initial looping.
      if (autoRefreshDone === prevAutoRefreshDone || prevAutoRefreshDone === null) {
        var _autoRefreshDone;
        // if this function was set and is executed, another refresh fetch can be triggered
        (_autoRefreshDone = autoRefreshDone) === null || _autoRefreshDone === void 0 ? void 0 : _autoRefreshDone();
        autoRefreshDone = undefined;
      }
    })).subscribe();
    return () => {
      if (numberOfFetches > 0) {
        subscription.unsubscribe();
      } else {
        // to let the initial fetch to execute properly before unsubscribing
        setTimeout(() => subscription.unsubscribe(), 200);
      }
    };
  }
  const fetchQuery = async () => {
    const query = appStateContainer.getState().query;
    const currentDataView = savedSearchContainer.getState().searchSource.getField('index');
    if ((0, _esQuery.isOfAggregateQueryType)(query)) {
      const nextDataView = await (0, _get_esql_data_view.getEsqlDataView)(query, currentDataView, services);
      if (nextDataView !== currentDataView) {
        setDataView(nextDataView);
      }
    }
    refetch$.next(undefined);
    return refetch$;
  };
  const fetchMore = () => {
    refetch$.next('fetch_more');
    return refetch$;
  };
  const reset = () => {
    (0, _use_saved_search_messages.sendResetMsg)(dataSubjects, getInitialFetchStatus());
  };
  const cancel = () => {
    var _abortController2, _abortControllerFetch2;
    (_abortController2 = abortController) === null || _abortController2 === void 0 ? void 0 : _abortController2.abort();
    (_abortControllerFetch2 = abortControllerFetchMore) === null || _abortControllerFetch2 === void 0 ? void 0 : _abortControllerFetch2.abort();
  };
  const getAbortController = () => {
    return abortController;
  };
  return {
    fetch: fetchQuery,
    fetchMore,
    data$: dataSubjects,
    refetch$,
    fetchChart$,
    disableNextFetchOnStateChange$,
    subscribe,
    reset,
    inspectorAdapters,
    getInitialFetchStatus,
    cancel,
    getAbortController
  };
}