"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processDocumentCountStats = exports.getDocumentCountStats = void 0;
var _lodash = require("lodash");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _seedrandom = _interopRequireDefault(require("seedrandom"));
var _mlIsDefined = require("@kbn/ml-is-defined");
var _random_sampler = require("../../constants/random_sampler");
var _build_query_filters = require("../../../../../common/utils/build_query_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MINIMUM_RANDOM_SAMPLER_DOC_COUNT = 100000;
const DEFAULT_INITIAL_RANDOM_SAMPLER_PROBABILITY = 0.000001;
const getDocumentCountStats = async (search, params, searchOptions, browserSessionSeed, probability, minimumRandomSamplerDocCount) => {
  var _firstResp$rawRespons3, _firstResp$rawRespons4;
  const seed = browserSessionSeed !== null && browserSessionSeed !== void 0 ? browserSessionSeed : Math.abs((0, _seedrandom.default)().int32()).toString();
  const {
    index,
    timeFieldName,
    earliest: earliestMs,
    latest: latestMs,
    runtimeFieldMap,
    searchQuery,
    intervalMs
  } = params;

  // Probability = 1 represents no sampling
  const result = {
    randomlySampled: false,
    took: 0,
    totalCount: 0,
    probability: 1
  };
  const filterCriteria = (0, _build_query_filters.buildFilterCriteria)(timeFieldName, earliestMs, latestMs, searchQuery);
  const query = {
    bool: {
      filter: filterCriteria
    }
  };
  // Don't use the sampler aggregation as this can lead to some potentially
  // confusing date histogram results depending on the date range of data amongst shards.
  const aggs = {
    eventRate: {
      date_histogram: {
        field: timeFieldName,
        fixed_interval: `${intervalMs}ms`,
        min_doc_count: 0,
        extended_bounds: {
          min: earliestMs,
          max: latestMs
        }
      }
    }
  };

  // If probability is provided, use that
  // Else, make an initial query using very low p
  // so that we can calculate the next p value that's appropriate for the data set
  const initialDefaultProbability = probability !== null && probability !== void 0 ? probability : DEFAULT_INITIAL_RANDOM_SAMPLER_PROBABILITY;
  const getAggsWithRandomSampling = p => ({
    sampler: {
      aggs,
      random_sampler: {
        probability: p,
        seed
      }
    }
  });
  const hasTimeField = timeFieldName !== undefined && timeFieldName !== '' && intervalMs !== undefined && intervalMs > 0;
  const getSearchParams = (aggregations, trackTotalHits = false) => ({
    index,
    query,
    ...(hasTimeField ? {
      aggs: aggregations
    } : {}),
    ...((0, _mlIsPopulatedObject.isPopulatedObject)(runtimeFieldMap) ? {
      runtime_mappings: runtimeFieldMap
    } : {}),
    track_total_hits: trackTotalHits,
    size: 0
  });
  const firstResp = await search.search({
    params: getSearchParams(getAggsWithRandomSampling(initialDefaultProbability),
    // Track total hits if time field is not defined
    !hasTimeField)
  }, searchOptions).toPromise();
  if (firstResp === undefined) {
    throw Error(`An error occurred with the following query ${JSON.stringify(getSearchParams(getAggsWithRandomSampling(initialDefaultProbability)))}`);
  }

  // If time field is not defined, no need to show the document count chart
  // Just need to return the tracked total hits
  if (!hasTimeField) {
    var _firstResp$rawRespons;
    const trackedTotalHits = typeof firstResp.rawResponse.hits.total === 'number' ? firstResp.rawResponse.hits.total : (_firstResp$rawRespons = firstResp.rawResponse.hits.total) === null || _firstResp$rawRespons === void 0 ? void 0 : _firstResp$rawRespons.value;
    return {
      ...result,
      randomlySampled: false,
      took: firstResp.rawResponse.took,
      totalCount: trackedTotalHits !== null && trackedTotalHits !== void 0 ? trackedTotalHits : 0
    };
  }
  if ((0, _mlIsDefined.isDefined)(probability)) {
    var _firstResp$rawRespons2;
    return {
      ...result,
      randomlySampled: probability === 1 ? false : true,
      took: (_firstResp$rawRespons2 = firstResp.rawResponse) === null || _firstResp$rawRespons2 === void 0 ? void 0 : _firstResp$rawRespons2.took,
      probability,
      ...processDocumentCountStats(firstResp.rawResponse, params, true)
    };
  }

  // @ts-expect-error ES types needs to be updated with doc_count as part random sampler aggregation
  const numSampled = (_firstResp$rawRespons3 = firstResp.rawResponse.aggregations) === null || _firstResp$rawRespons3 === void 0 ? void 0 : (_firstResp$rawRespons4 = _firstResp$rawRespons3.sampler) === null || _firstResp$rawRespons4 === void 0 ? void 0 : _firstResp$rawRespons4.doc_count;
  const numDocs = minimumRandomSamplerDocCount !== null && minimumRandomSamplerDocCount !== void 0 ? minimumRandomSamplerDocCount : MINIMUM_RANDOM_SAMPLER_DOC_COUNT;
  if (firstResp !== undefined && numSampled < numDocs) {
    const newProbability = initialDefaultProbability * numDocs / (numSampled - 2 * Math.sqrt(numSampled));

    // If the number of docs is < 3 million
    // proceed to make a vanilla aggregation without any sampling (probability = 1)
    // Minimum of 4 docs (3e6 * 0.000001 + 1) sampled gives us 90% confidence interval # docs is within
    if (newProbability === Infinity || numSampled <= 4) {
      var _vanillaAggResp$rawRe, _vanillaAggResp$rawRe2;
      const vanillaAggResp = await search.search({
        params: getSearchParams(getAggsWithRandomSampling(1))
      }, searchOptions).toPromise();
      return {
        ...result,
        randomlySampled: false,
        took: firstResp.rawResponse.took + ((_vanillaAggResp$rawRe = vanillaAggResp === null || vanillaAggResp === void 0 ? void 0 : (_vanillaAggResp$rawRe2 = vanillaAggResp.rawResponse) === null || _vanillaAggResp$rawRe2 === void 0 ? void 0 : _vanillaAggResp$rawRe2.took) !== null && _vanillaAggResp$rawRe !== void 0 ? _vanillaAggResp$rawRe : 0),
        ...processDocumentCountStats(vanillaAggResp === null || vanillaAggResp === void 0 ? void 0 : vanillaAggResp.rawResponse, params, true),
        probability: 1
      };
    } else {
      // Else, make second random sampler
      const closestProbability = _random_sampler.RANDOM_SAMPLER_PROBABILITIES[(0, _lodash.sortedIndex)(_random_sampler.RANDOM_SAMPLER_PROBABILITIES, newProbability)];
      const secondResp = await search.search({
        params: getSearchParams(getAggsWithRandomSampling(closestProbability))
      }, searchOptions).toPromise();
      if (secondResp) {
        return {
          ...result,
          randomlySampled: true,
          took: firstResp.rawResponse.took + secondResp.rawResponse.took,
          ...processDocumentCountStats(secondResp.rawResponse, params, true),
          probability: closestProbability
        };
      }
    }
  }
  return result;
};
exports.getDocumentCountStats = getDocumentCountStats;
const processDocumentCountStats = (body, params, randomlySampled = false) => {
  if (!body) return undefined;
  let totalCount = 0;
  if (params.intervalMs === undefined || params.earliest === undefined || params.latest === undefined) {
    return {
      totalCount
    };
  }
  const buckets = {};
  const dataByTimeBucket = (0, _lodash.get)(body, randomlySampled ? ['aggregations', 'sampler', 'eventRate', 'buckets'] : ['aggregations', 'eventRate', 'buckets'], []);
  (0, _lodash.each)(dataByTimeBucket, dataForTime => {
    const time = dataForTime.key;
    buckets[time] = dataForTime.doc_count;
    totalCount += dataForTime.doc_count;
  });
  return {
    interval: params.intervalMs,
    buckets,
    timeRangeEarliest: typeof params.earliest === 'number' ? params.earliest : undefined,
    timeRangeLatest: typeof params.latest === 'number' ? params.latest : undefined,
    totalCount
  };
};
exports.processDocumentCountStats = processDocumentCountStats;