"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.schema = void 0;
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/data-views-plugin/common");
var _shared_imports = require("../../shared_imports");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const {
  containsCharsField,
  emptyField,
  numberGreaterThanField,
  maxLengthField
} = _shared_imports.fieldValidators;
const i18nTexts = {
  invalidScriptErrorMessage: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.scriptEditorPainlessValidationMessage', {
    defaultMessage: 'Invalid Painless script.'
  })
};

// Validate the painless **script**
const painlessScriptValidator = async ({
  customData: {
    provider
  }
}) => {
  const previewError = await provider();
  if (previewError && previewError.code === 'PAINLESS_SCRIPT_ERROR') {
    return {
      message: i18nTexts.invalidScriptErrorMessage
    };
  }
};
const schema = exports.schema = {
  name: {
    label: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.nameLabel', {
      defaultMessage: 'Name'
    }),
    validations: [{
      validator: emptyField(_i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.nameIsRequiredErrorMessage', {
        defaultMessage: 'A name is required.'
      }))
    }, {
      validator: containsCharsField({
        message: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.starCharacterNotAllowedValidationErrorMessage', {
          defaultMessage: 'The field cannot have * in the name.'
        }),
        chars: '*'
      })
    }]
  },
  type: {
    label: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.runtimeTypeLabel', {
      defaultMessage: 'Type'
    }),
    defaultValue: [_constants.RUNTIME_FIELD_OPTIONS[0]],
    fieldsToValidateOnChange: ['script.source']
  },
  script: {
    source: {
      label: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.defineFieldLabel', {
        defaultMessage: 'Define script'
      }),
      validations: [{
        validator: emptyField(_i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.scriptIsRequiredErrorMessage', {
          defaultMessage: 'A script is required to set the field value.'
        }))
      }, {
        validator: painlessScriptValidator,
        isAsync: true
      }]
    }
  },
  customLabel: {
    label: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.customLabelLabel', {
      defaultMessage: 'Custom label'
    }),
    validations: [{
      validator: emptyField(_i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.customLabelIsRequiredErrorMessage', {
        defaultMessage: 'Give a label to the field.'
      }))
    }]
  },
  customDescription: {
    label: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.customDescriptionLabel', {
      defaultMessage: 'Custom description'
    }),
    validations: [{
      validator: emptyField(_i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.customDescriptionIsRequiredErrorMessage', {
        defaultMessage: 'Give a description to the field.'
      }))
    }, {
      validator: maxLengthField({
        length: _common.MAX_DATA_VIEW_FIELD_DESCRIPTION_LENGTH,
        message: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.customDescriptionMaxLengthErrorMessage', {
          values: {
            length: _common.MAX_DATA_VIEW_FIELD_DESCRIPTION_LENGTH
          },
          defaultMessage: 'The length of the description is too long. The maximum length is {length} characters.'
        })
      })
    }]
  },
  popularity: {
    label: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.popularityLabel', {
      defaultMessage: 'Popularity'
    }),
    validations: [{
      validator: emptyField(_i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.popularityIsRequiredErrorMessage', {
        defaultMessage: 'Give a popularity to the field.'
      }))
    }, {
      validator: numberGreaterThanField({
        than: 0,
        allowEquality: true,
        message: _i18n.i18n.translate('indexPatternFieldEditor.editor.form.validations.popularityGreaterThan0ErrorMessage', {
          defaultMessage: 'The popularity must be zero or greater.'
        })
      })
    }]
  },
  fields: {
    defaultValue: {}
  },
  __meta__: {
    isCustomLabelVisible: {
      defaultValue: false
    },
    isCustomDescriptionVisible: {
      defaultValue: false
    },
    isValueVisible: {
      defaultValue: false
    },
    isFormatVisible: {
      defaultValue: false
    },
    isPopularityVisible: {
      defaultValue: false
    }
  }
};