"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SEARCH_API_BASE_URL = void 0;
exports.registerSearchRoute = registerSearchRoute;
var _rxjs = require("rxjs");
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/kibana-utils-plugin/server");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _report_search_error = require("../report_search_error");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const SEARCH_API_BASE_URL = exports.SEARCH_API_BASE_URL = '/internal/search';
function registerSearchRoute(router, logger, executionContextSetup) {
  router.versioned.post({
    path: `${SEARCH_API_BASE_URL}/{strategy}/{id?}`,
    access: 'internal',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _configSchema.schema.object({
          strategy: _configSchema.schema.string(),
          id: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        body: _configSchema.schema.object({
          legacyHitsTotal: _configSchema.schema.maybe(_configSchema.schema.boolean()),
          sessionId: _configSchema.schema.maybe(_configSchema.schema.string()),
          isStored: _configSchema.schema.maybe(_configSchema.schema.boolean()),
          isRestore: _configSchema.schema.maybe(_configSchema.schema.boolean()),
          retrieveResults: _configSchema.schema.maybe(_configSchema.schema.boolean()),
          stream: _configSchema.schema.maybe(_configSchema.schema.boolean())
        }, {
          unknowns: 'allow'
        })
      }
    }
  }, async (context, request, res) => {
    const {
      legacyHitsTotal = true,
      sessionId,
      isStored,
      isRestore,
      retrieveResults,
      stream,
      ...searchRequest
    } = request.body;
    const {
      strategy,
      id
    } = request.params;
    const abortSignal = (0, _lib.getRequestAbortedSignal)(request.events.aborted$);
    let executionContext;
    const contextHeader = request.headers['x-kbn-context'];
    try {
      if (contextHeader != null) {
        executionContext = JSON.parse(decodeURIComponent(Array.isArray(contextHeader) ? contextHeader[0] : contextHeader));
      }
    } catch (err) {
      logger.error(`Error parsing search execution context: ${contextHeader}`);
    }
    return executionContextSetup.withContext(executionContext, async () => {
      _elasticApmNode.default.addLabels(executionContextSetup.getAsLabels());
      try {
        const search = await context.search;
        const response = await search.search({
          ...searchRequest,
          id
        }, {
          abortSignal,
          strategy,
          legacyHitsTotal,
          sessionId,
          isStored,
          isRestore,
          retrieveResults,
          stream
        }).pipe((0, _rxjs.first)()).toPromise();
        if (response && response.rawResponse.pipe) {
          return res.ok({
            body: response.rawResponse,
            headers: {
              'kbn-search-is-restored': response.isRestored ? '?1' : '?0',
              'kbn-search-request-params': JSON.stringify(response.requestParams)
            }
          });
        } else {
          return res.ok({
            body: response
          });
        }
      } catch (err) {
        return (0, _report_search_error.reportSearchError)(res, err);
      }
    });
  });
  router.versioned.delete({
    path: '/internal/search/{strategy}/{id}',
    access: 'internal',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _configSchema.schema.object({
          strategy: _configSchema.schema.string(),
          id: _configSchema.schema.string()
        })
      }
    }
  }, async (context, request, res) => {
    const {
      strategy,
      id
    } = request.params;
    try {
      const search = await context.search;
      await search.cancel(id, {
        strategy
      });
      return res.ok();
    } catch (err) {
      return (0, _server.reportServerError)(res, err);
    }
  });
}