"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ApmSystem = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _std = require("@kbn/std");
var _ebt_span_filter = require("./filters/ebt_span_filter");
var _apm_resource_counter = require("./apm_resource_counter");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/** "GET protocol://hostname:port/pathname" */
const HTTP_REQUEST_TRANSACTION_NAME_REGEX = /^(GET|POST|PUT|HEAD|PATCH|DELETE|OPTIONS|CONNECT|TRACE)\s(.*)$/;
const USER_INTERACTION_ATTR_NAME = 'data-test-subj';

/**
 * This is the entry point used to boot the frontend when serving a application
 * that lives in the Kibana Platform.
 */

class ApmSystem {
  /**
   * `apmConfig` would be populated with relevant APM RUM agent
   * configuration if server is started with elastic.apm.* config.
   */
  constructor(apmConfig, basePath = '') {
    (0, _defineProperty2.default)(this, "enabled", void 0);
    (0, _defineProperty2.default)(this, "pageLoadTransaction", void 0);
    (0, _defineProperty2.default)(this, "resourceObserver", void 0);
    (0, _defineProperty2.default)(this, "apm", void 0);
    (0, _defineProperty2.default)(this, "executionContext", void 0);
    this.apmConfig = apmConfig;
    this.basePath = basePath;
    this.enabled = apmConfig != null && !!apmConfig.active;
    this.resourceObserver = new _apm_resource_counter.CachedResourceObserver();
  }
  async setup() {
    if (!this.enabled) return;
    const {
      init,
      apm
    } = await Promise.resolve().then(() => _interopRequireWildcard(require('@elastic/apm-rum')));
    this.apm = apm;
    const {
      globalLabels,
      ...apmConfig
    } = this.apmConfig;
    if (globalLabels) {
      apm.addLabels(globalLabels);
    }
    apm.addLabels({
      user_agent: navigator.userAgent
    });
    apm.addFilter(_ebt_span_filter.ebtSpanFilter);
    this.addHttpRequestNormalization(apm);
    this.addRouteChangeNormalization(apm);
    init(apmConfig);
    // hold page load transaction blocks a transaction implicitly created by init.
    this.holdPageLoadTransaction(apm);

    // Registering an event handler to improve user-interaction transaction names based on data-test-subj attribute
    // Context: https://github.com/elastic/observability-dev/issues/4528
    window.addEventListener('click', function (event) {
      const tr = apm.getCurrentTransaction();
      if (!tr) {
        // In some cases agent decides not to create a transaction
        return;
      }
      const {
        target
      } = event;
      if (target instanceof Element) {
        const element = target === null || target === void 0 ? void 0 : target.closest(`a[${USER_INTERACTION_ATTR_NAME}], button[${USER_INTERACTION_ATTR_NAME}]`);
        if (element) {
          tr.name = `Click - ${element.getAttribute(USER_INTERACTION_ATTR_NAME)}`;
        } else if (target.getAttribute(USER_INTERACTION_ATTR_NAME)) {
          tr.name = `Click - ${target.getAttribute(USER_INTERACTION_ATTR_NAME)}`;
        }
      }
    }, true);
  }
  async start(start) {
    if (!this.enabled || !start) return;
    this.executionContext = start.executionContext;
    this.markPageLoadStart();
    start.executionContext.context$.subscribe(c => {
      var _this$apm;
      // We're using labels because we want the context to be indexed
      // https://www.elastic.co/guide/en/apm/get-started/current/metadata.html
      const apmContext = start.executionContext.getAsLabels();
      (_this$apm = this.apm) === null || _this$apm === void 0 ? void 0 : _this$apm.addLabels(apmContext);
    });

    // TODO: Start a new transaction every page change instead of every app change.

    /**
     * Register listeners for navigation changes and capture them as
     * route-change transactions after Kibana app is bootstrapped
     */
    start.application.currentAppId$.subscribe(appId => {
      if (appId && this.apm) {
        this.closePageLoadTransaction();
        this.apm.startTransaction(appId, 'app-change', {
          managed: true,
          canReuse: true
        });
      }
    });
  }

  /* Hold the page load transaction open, until all resources actually finish loading */
  holdPageLoadTransaction(apm) {
    const transaction = apm.getCurrentTransaction();

    // Keep the page load transaction open until all resources finished loading
    if (transaction && transaction.type === 'page-load') {
      this.pageLoadTransaction = transaction;
      // @ts-expect-error 2339  block is a private property of Transaction interface
      this.pageLoadTransaction.block(true);
      this.pageLoadTransaction.mark('apm-setup');
    }
  }

  /* Close and clear the page load transaction */
  closePageLoadTransaction() {
    if (this.pageLoadTransaction) {
      const loadCounts = this.resourceObserver.getCounts();
      this.pageLoadTransaction.addLabels({
        'loaded-resources': loadCounts.networkOrDisk,
        'cached-resources': loadCounts.memory
      });
      this.resourceObserver.destroy();
      this.pageLoadTransaction.end();
      this.pageLoadTransaction = undefined;
    }
  }
  markPageLoadStart() {
    if (this.pageLoadTransaction) {
      this.pageLoadTransaction.mark('apm-start');
    }
  }

  /**
   * Adds an observer to the APM configuration for normalizing transactions of the 'http-request' type to remove the
   * hostname, protocol, port, and base path. Allows for correlating data cross different deployments.
   */
  addHttpRequestNormalization(apm) {
    apm.observe('transaction:end', t => {
      if (t.type !== 'http-request') {
        return;
      }

      /** Split URLs of the from "GET protocol://hostname:port/pathname" into method & hostname */
      const matches = t.name.match(HTTP_REQUEST_TRANSACTION_NAME_REGEX);
      if (!matches) {
        return;
      }
      const [, method, originalUrl] = matches;
      // Normalize the URL
      const normalizedUrl = (0, _std.modifyUrl)(originalUrl, parts => {
        var _parts$pathname;
        const isAbsolute = parts.hostname && parts.protocol && parts.port;
        // If the request was to a different host, port, or protocol then don't change anything
        if (isAbsolute && (parts.hostname !== window.location.hostname || parts.protocol !== window.location.protocol || parts.port !== window.location.port)) {
          return;
        }

        // Strip the protocol, hostname, port, and protocol slashes to normalize
        parts.protocol = null;
        parts.hostname = null;
        parts.port = null;
        parts.slashes = false;

        // Replace the basePath if present in the pathname
        if (parts.pathname === this.basePath) {
          parts.pathname = '/';
        } else if ((_parts$pathname = parts.pathname) !== null && _parts$pathname !== void 0 && _parts$pathname.startsWith(`${this.basePath}/`)) {
          var _parts$pathname2;
          parts.pathname = (_parts$pathname2 = parts.pathname) === null || _parts$pathname2 === void 0 ? void 0 : _parts$pathname2.slice(this.basePath.length);
        }
      });
      t.name = `${method} ${normalizedUrl}`;
    });
  }

  /**
   * Set route-change transaction name to the destination page name taken from
   * the execution context. Otherwise, all route change transactions would have
   * default names, like 'Click - span' or 'Click - a' instead of more
   * descriptive '/security/rules/:id/edit'.
   */
  addRouteChangeNormalization(apm) {
    apm.observe('transaction:end', t => {
      var _this$executionContex;
      const executionContext = (_this$executionContex = this.executionContext) === null || _this$executionContex === void 0 ? void 0 : _this$executionContex.get();
      if (executionContext && t.type === 'route-change') {
        const {
          name,
          page
        } = executionContext;
        t.name = `${name} ${page || 'unknown'}`;
      }
    });
  }
}
exports.ApmSystem = ApmSystem;