"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PluginWrapper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _path = require("path");
var _typeDetect = _interopRequireDefault(require("type-detect"));
var _rxjs = require("rxjs");
var _std = require("@kbn/std");
var _configSchema = require("@kbn/config-schema");
var _coreBaseCommon = require("@kbn/core-base-common");
var _coreDi = require("@kbn/core-di");
var _coreDiInternal = require("@kbn/core-di-internal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const OSS_PATH_REGEX = /[\/|\\]src[\/|\\]plugins[\/|\\]/; // Matches src/plugins directory on POSIX and Windows
const XPACK_PATH_REGEX = /[\/|\\]x-pack[\/|\\]plugins[\/|\\]/; // Matches x-pack/plugins directory on POSIX and Windows

/**
 * Lightweight wrapper around discovered plugin that is responsible for instantiating
 * plugin and dispatching proper context and dependencies into plugin's lifecycle hooks.
 *
 * @internal
 */
class PluginWrapper {
  constructor(params) {
    (0, _defineProperty2.default)(this, "path", void 0);
    (0, _defineProperty2.default)(this, "source", void 0);
    (0, _defineProperty2.default)(this, "manifest", void 0);
    (0, _defineProperty2.default)(this, "opaqueId", void 0);
    (0, _defineProperty2.default)(this, "name", void 0);
    (0, _defineProperty2.default)(this, "configPath", void 0);
    (0, _defineProperty2.default)(this, "requiredPlugins", void 0);
    (0, _defineProperty2.default)(this, "optionalPlugins", void 0);
    (0, _defineProperty2.default)(this, "runtimePluginDependencies", void 0);
    (0, _defineProperty2.default)(this, "requiredBundles", void 0);
    (0, _defineProperty2.default)(this, "includesServerPlugin", void 0);
    (0, _defineProperty2.default)(this, "includesUiPlugin", void 0);
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "initializerContext", void 0);
    (0, _defineProperty2.default)(this, "definition", void 0);
    (0, _defineProperty2.default)(this, "instance", void 0);
    (0, _defineProperty2.default)(this, "container", void 0);
    (0, _defineProperty2.default)(this, "startDependencies$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "startDependencies", (0, _rxjs.firstValueFrom)(this.startDependencies$));
    this.params = params;
    this.path = params.path;
    this.source = getPluginSource(params.path);
    this.manifest = params.manifest;
    this.opaqueId = params.opaqueId;
    this.initializerContext = params.initializerContext;
    this.log = params.initializerContext.logger.get();
    this.name = params.manifest.id;
    this.configPath = params.manifest.configPath;
    this.requiredPlugins = params.manifest.requiredPlugins;
    this.optionalPlugins = params.manifest.optionalPlugins;
    this.requiredBundles = params.manifest.requiredBundles;
    this.runtimePluginDependencies = params.manifest.runtimePluginDependencies;
    this.includesServerPlugin = params.manifest.server;
    this.includesUiPlugin = params.manifest.ui;
  }
  async init() {
    this.log.debug('Initializing plugin');
    this.definition = this.getPluginDefinition();
    this.instance = await this.createPluginInstance();
    if (!('plugin' in this.definition || 'module' in this.definition)) {
      throw new Error(`Plugin "${this.name}" does not export the "plugin" definition or "module" (${this.path}).`);
    }
  }

  /**
   * Instantiates plugin and calls `setup` function exposed by the plugin initializer.
   * @param setupContext Context that consists of various core services tailored specifically
   * for the `setup` lifecycle event.
   * @param plugins The dictionary where the key is the dependency name and the value
   * is the contract returned by the dependency's `setup` function.
   */
  setup(setupContext, plugins) {
    var _this$instance, _this$container;
    if (!this.definition) {
      throw new Error('The plugin is not initialized. Call the init method first.');
    }
    if (this.instance && this.isPrebootPluginInstance(this.instance)) {
      return this.instance.setup(setupContext, plugins);
    }
    if (this.definition.module) {
      this.container = setupContext.injection.getContainer();
      this.container.loadSync(this.definition.module);
      this.container.loadSync((0, _coreDiInternal.createSetupModule)(this.initializerContext, setupContext, plugins));
    }
    return [(_this$instance = this.instance) === null || _this$instance === void 0 ? void 0 : _this$instance.setup(setupContext, plugins), (_this$container = this.container) === null || _this$container === void 0 ? void 0 : _this$container.get(_coreDi.Setup)].find(Boolean);
  }

  /**
   * Calls `start` function exposed by the initialized plugin.
   * @param startContext Context that consists of various core services tailored specifically
   * for the `start` lifecycle event.
   * @param plugins The dictionary where the key is the dependency name and the value
   * is the contract returned by the dependency's `start` function.
   */
  start(startContext, plugins) {
    var _this$container2, _this$instance2, _this$container3;
    if (!this.definition) {
      throw new Error(`Plugin "${this.name}" can't be started since it isn't set up.`);
    }
    if (this.instance && this.isPrebootPluginInstance(this.instance)) {
      throw new Error(`Plugin "${this.name}" is a preboot plugin and cannot be started.`);
    }
    (_this$container2 = this.container) === null || _this$container2 === void 0 ? void 0 : _this$container2.loadSync((0, _coreDiInternal.createStartModule)(startContext, plugins));
    const contract = [(_this$instance2 = this.instance) === null || _this$instance2 === void 0 ? void 0 : _this$instance2.start(startContext, plugins), (_this$container3 = this.container) === null || _this$container3 === void 0 ? void 0 : _this$container3.get(_coreDi.Start)].find(Boolean);
    if ((0, _std.isPromise)(contract)) {
      return contract.then(resolvedContract => {
        this.startDependencies$.next([startContext, plugins, resolvedContract]);
        return resolvedContract;
      });
    }
    this.startDependencies$.next([startContext, plugins, contract]);
    return contract;
  }

  /**
   * Calls optional `stop` function exposed by the plugin initializer.
   */
  async stop() {
    var _this$instance3, _this$instance3$stop, _this$container4;
    if (!this.definition) {
      throw new Error(`Plugin "${this.name}" can't be stopped since it isn't set up.`);
    }
    await ((_this$instance3 = this.instance) === null || _this$instance3 === void 0 ? void 0 : (_this$instance3$stop = _this$instance3.stop) === null || _this$instance3$stop === void 0 ? void 0 : _this$instance3$stop.call(_this$instance3));
    await ((_this$container4 = this.container) === null || _this$container4 === void 0 ? void 0 : _this$container4.unbindAll());
    this.instance = undefined;
    this.container = undefined;
  }
  getConfigDescriptor() {
    if (!this.manifest.server) {
      return null;
    }
    const definition = this.getPluginDefinition();
    if (!definition.config) {
      this.log.debug(`Plugin "${this.name}" does not export "config" (${this.path}).`);
      return null;
    }
    const {
      config
    } = definition;
    if (!(0, _configSchema.isConfigSchema)(config.schema)) {
      throw new Error('Configuration schema expected to be an instance of Type');
    }
    return config;
  }
  getPluginDefinition() {
    var _require;
    return (_require = require((0, _path.join)(this.path, 'server'))) !== null && _require !== void 0 ? _require : {};
  }
  async createPluginInstance() {
    var _this$definition;
    if (!((_this$definition = this.definition) !== null && _this$definition !== void 0 && _this$definition.plugin)) {
      return;
    }
    const {
      plugin: initializer
    } = this.definition;
    if (!initializer || typeof initializer !== 'function') {
      throw new Error(`Definition of plugin "${this.name}" should be a function (${this.path}).`);
    }
    const instance = await initializer(this.initializerContext);
    if (!instance || typeof instance !== 'object') {
      throw new Error(`Initializer for plugin "${this.manifest.id}" is expected to return plugin instance, but returned "${(0, _typeDetect.default)(instance)}".`);
    }
    if (typeof instance.setup !== 'function') {
      throw new Error(`Instance of plugin "${this.name}" does not define "setup" function.`);
    }
    return instance;
  }
  isPrebootPluginInstance(instance) {
    return this.manifest.type === _coreBaseCommon.PluginType.preboot;
  }
}
exports.PluginWrapper = PluginWrapper;
function getPluginSource(path) {
  if (OSS_PATH_REGEX.test(path)) {
    return 'oss';
  } else if (XPACK_PATH_REGEX.test(path)) {
    return 'x-pack';
  }
  return 'external';
}