"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatErrorMeta = formatErrorMeta;
exports.getRouteFullPath = getRouteFullPath;
exports.getVersionHeader = getVersionHeader;
exports.injectResponseHeaders = injectResponseHeaders;
exports.injectVersionHeader = injectVersionHeader;
exports.isSafeMethod = isSafeMethod;
exports.prepareResponseValidation = prepareResponseValidation;
exports.prepareRouteConfigValidation = prepareRouteConfigValidation;
exports.validOptions = validOptions;
var _lodash = require("lodash");
var _coreHttpServer = require("@kbn/core-http-server");
var _coreHttpCommon = require("@kbn/core-http-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isStatusCode(key) {
  return !isNaN(parseInt(key, 10));
}
function prepareResponseValidation(validation) {
  const responses = Object.entries(validation).map(([key, value]) => {
    if (isStatusCode(key)) {
      return [key, {
        ...value,
        ...(value.body ? {
          body: (0, _lodash.once)(value.body)
        } : {})
      }];
    }
    return [key, value];
  });
  return Object.fromEntries(responses);
}
function prepareValidation(validator) {
  if ((0, _coreHttpServer.isFullValidatorContainer)(validator) && validator.response) {
    return {
      ...validator,
      response: prepareResponseValidation(validator.response)
    };
  }
  return validator;
}

// Integration tested in ./routes.test.ts
function prepareRouteConfigValidation(config) {
  // Calculating schema validation can be expensive so when it is provided lazily
  // we only want to instantiate it once. This also provides idempotency guarantees
  if (typeof config.validate === 'function') {
    const validate = config.validate;
    return {
      ...config,
      validate: (0, _lodash.once)(() => prepareValidation(validate()))
    };
  } else if (typeof config.validate === 'object' && typeof config.validate !== null) {
    return {
      ...config,
      validate: prepareValidation(config.validate)
    };
  }
  return config;
}

/**
 * @note mutates the response object
 * @internal
 */
function injectResponseHeaders(headers, response) {
  const mutableResponse = response;
  mutableResponse.options.headers = {
    ...mutableResponse.options.headers,
    ...headers
  };
  return mutableResponse;
}
function getVersionHeader(version) {
  return {
    [_coreHttpCommon.ELASTIC_HTTP_VERSION_HEADER]: version
  };
}
function injectVersionHeader(version, response) {
  return injectResponseHeaders(getVersionHeader(version), response);
}
function formatErrorMeta(statusCode, {
  error,
  request
}) {
  var _request$route, _request$route2;
  return {
    http: {
      response: {
        status_code: statusCode
      },
      request: {
        method: (_request$route = request.route) === null || _request$route === void 0 ? void 0 : _request$route.method,
        path: (_request$route2 = request.route) === null || _request$route2 === void 0 ? void 0 : _request$route2.path
      }
    },
    error: {
      message: error.message
    }
  };
}
function getRouteFullPath(routerPath, routePath) {
  // If router's path ends with slash and route's path starts with slash,
  // we should omit one of them to have a valid concatenated path.
  const routePathStartIndex = routerPath.endsWith('/') && routePath.startsWith('/') ? 1 : 0;
  return `${routerPath}${routePath.slice(routePathStartIndex)}`;
}
function isSafeMethod(method) {
  return method === 'get' || method === 'options';
}

/**
 * Create a valid options object with "sensible" defaults + adding some validation to the options fields
 *
 * @param method HTTP verb for these options
 * @param routeConfig The route config definition
 */
function validOptions(method, routeConfig) {
  const shouldNotHavePayload = ['head', 'get'].includes(method);
  const {
    options = {},
    validate
  } = routeConfig;
  const shouldValidateBody = validate && !!(0, _coreHttpServer.getRequestValidation)(validate).body || !!options.body;
  const {
    output
  } = options.body || {};
  if (typeof output === 'string' && !_coreHttpServer.validBodyOutput.includes(output)) {
    throw new Error(`[options.body.output: '${output}'] in route ${method.toUpperCase()} ${routeConfig.path} is not valid. Only '${_coreHttpServer.validBodyOutput.join("' or '")}' are valid.`);
  }

  // @ts-expect-error to eliminate problems with `security` in the options for route factories abstractions
  if (options.security) {
    throw new Error('`options.security` is not allowed in route config. Use `security` instead.');
  }
  const body = shouldNotHavePayload ? undefined : {
    // If it's not a GET (requires payload) but no body validation is required (or no body options are specified),
    // We assume the route does not care about the body => use the memory-cheapest approach (stream and no parsing)
    output: !shouldValidateBody ? 'stream' : undefined,
    parse: !shouldValidateBody ? false : undefined,
    // User's settings should overwrite any of the "desired" values
    ...options.body
  };
  return {
    ...options,
    body
  };
}