"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AbstractLogger = void 0;
var _logging = require("@kbn/logging");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * @internal
 */

/**
 * A basic, abstract logger implementation that delegates the create of log records to the child's createLogRecord function.
 * @internal
 */
class AbstractLogger {
  constructor(context, level, appenders, factory) {
    this.context = context;
    this.level = level;
    this.appenders = appenders;
    this.factory = factory;
  }
  trace(message, meta) {
    if (!this.level.supports(_logging.LogLevel.Trace)) {
      return;
    }
    if (typeof message === 'function') {
      message = message();
    }
    this.performLog(this.createLogRecord(_logging.LogLevel.Trace, message, meta));
  }
  debug(message, meta) {
    if (!this.level.supports(_logging.LogLevel.Debug)) {
      return;
    }
    if (typeof message === 'function') {
      message = message();
    }
    this.performLog(this.createLogRecord(_logging.LogLevel.Debug, message, meta));
  }
  info(message, meta) {
    if (!this.level.supports(_logging.LogLevel.Info)) {
      return;
    }
    if (typeof message === 'function') {
      message = message();
    }
    this.performLog(this.createLogRecord(_logging.LogLevel.Info, message, meta));
  }
  warn(errorOrMessage, meta) {
    if (!this.level.supports(_logging.LogLevel.Warn)) {
      return;
    }
    if (typeof errorOrMessage === 'function') {
      errorOrMessage = errorOrMessage();
    }
    this.performLog(this.createLogRecord(_logging.LogLevel.Warn, errorOrMessage, meta));
  }
  error(errorOrMessage, meta) {
    if (!this.level.supports(_logging.LogLevel.Error)) {
      return;
    }
    if (typeof errorOrMessage === 'function') {
      errorOrMessage = errorOrMessage();
    }
    this.performLog(this.createLogRecord(_logging.LogLevel.Error, errorOrMessage, meta));
  }
  fatal(errorOrMessage, meta) {
    if (!this.level.supports(_logging.LogLevel.Fatal)) {
      return;
    }
    if (typeof errorOrMessage === 'function') {
      errorOrMessage = errorOrMessage();
    }
    this.performLog(this.createLogRecord(_logging.LogLevel.Fatal, errorOrMessage, meta));
  }
  isLevelEnabled(levelId) {
    return this.level.supports(_logging.LogLevel.fromId(levelId));
  }
  log(record) {
    if (!this.level.supports(record.level)) {
      return;
    }
    this.performLog(record);
  }
  get(...childContextPaths) {
    return this.factory.get(...[this.context, ...childContextPaths]);
  }
  performLog(record) {
    for (const appender of this.appenders) {
      appender.append(record);
    }
  }
}
exports.AbstractLogger = AbstractLogger;