"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ElasticsearchService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _coreElasticsearchClientServerInternal = require("@kbn/core-elasticsearch-client-server-internal");
var _register_analytics_context_provider = require("./register_analytics_context_provider");
var _elasticsearch_config = require("./elasticsearch_config");
var _ensure_es_version = require("./version_check/ensure_es_version");
var _status = require("./status");
var _is_valid_connection = require("./is_valid_connection");
var _is_scripting_enabled = require("./is_scripting_enabled");
var _merge_config = require("./merge_config");
var _get_cluster_info = require("./get_cluster_info");
var _get_capabilities = require("./get_capabilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** @internal */
class ElasticsearchService {
  constructor(coreContext) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "config$", void 0);
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "kibanaVersion", void 0);
    (0, _defineProperty2.default)(this, "authHeaders", void 0);
    (0, _defineProperty2.default)(this, "executionContextClient", void 0);
    (0, _defineProperty2.default)(this, "esNodesCompatibility$", void 0);
    (0, _defineProperty2.default)(this, "client", void 0);
    (0, _defineProperty2.default)(this, "clusterInfo$", void 0);
    (0, _defineProperty2.default)(this, "unauthorizedErrorHandler", void 0);
    (0, _defineProperty2.default)(this, "agentManager", void 0);
    this.coreContext = coreContext;
    this.kibanaVersion = coreContext.env.packageInfo.version;
    this.log = coreContext.logger.get('elasticsearch-service');
    this.config$ = coreContext.configService.atPath('elasticsearch').pipe((0, _rxjs.map)(rawConfig => new _elasticsearch_config.ElasticsearchConfig(rawConfig)));
  }
  async preboot() {
    this.log.debug('Prebooting elasticsearch service');
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    return {
      config: {
        hosts: config.hosts,
        credentialsSpecified: config.username !== undefined || config.password !== undefined || config.serviceAccountToken !== undefined
      },
      createClient: (type, clientConfig) => this.createClusterClient(type, config, clientConfig)
    };
  }
  async setup(deps) {
    this.log.debug('Setting up elasticsearch service');
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    const agentManager = this.getAgentManager(config);
    this.authHeaders = deps.http.authRequestHeaders;
    this.executionContextClient = deps.executionContext;
    this.client = this.createClusterClient('data', config);
    const esNodesCompatibility$ = (0, _ensure_es_version.pollEsNodesVersion)({
      kibanaVersion: this.kibanaVersion,
      ignoreVersionMismatch: config.ignoreVersionMismatch,
      healthCheckInterval: config.healthCheckDelay.asMilliseconds(),
      healthCheckStartupInterval: config.healthCheckStartupDelay.asMilliseconds(),
      log: this.log,
      internalClient: this.client.asInternalUser
    }).pipe((0, _rxjs.takeUntil)(this.stop$));

    // Log every error we may encounter in the connection to Elasticsearch
    esNodesCompatibility$.subscribe(({
      isCompatible,
      message
    }) => {
      if (!isCompatible && message) {
        this.log.error(message);
      }
    });
    this.esNodesCompatibility$ = esNodesCompatibility$;
    this.clusterInfo$ = (0, _get_cluster_info.getClusterInfo$)(this.client.asInternalUser).pipe((0, _rxjs.takeUntil)(this.stop$));
    (0, _register_analytics_context_provider.registerAnalyticsContextProvider)(deps.analytics, this.clusterInfo$);
    return {
      legacy: {
        config$: this.config$
      },
      clusterInfo$: this.clusterInfo$,
      esNodesCompatibility$,
      status$: (0, _status.calculateStatus$)(esNodesCompatibility$),
      setUnauthorizedErrorHandler: handler => {
        if (this.unauthorizedErrorHandler) {
          throw new Error('setUnauthorizedErrorHandler can only be called once.');
        }
        this.unauthorizedErrorHandler = handler;
      },
      agentStatsProvider: {
        getAgentsStats: agentManager.getAgentsStats.bind(agentManager)
      },
      publicBaseUrl: config.publicBaseUrl
    };
  }
  async start() {
    if (!this.client || !this.esNodesCompatibility$) {
      throw new Error('ElasticsearchService needs to be setup before calling start');
    }
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    let capabilities;
    let elasticsearchWaitTime;
    if (!config.skipStartupConnectionCheck) {
      const elasticsearchWaitStartTime = performance.now();
      // Ensure that the connection is established and the product is valid before moving on
      await (0, _is_valid_connection.isValidConnection)(this.esNodesCompatibility$);
      elasticsearchWaitTime = Math.round(performance.now() - elasticsearchWaitStartTime);
      this.log.info(`Successfully connected to Elasticsearch after waiting for ${elasticsearchWaitTime} milliseconds`, {
        event: {
          // ECS Event reference: https://www.elastic.co/docs/reference/ecs/ecs-event
          action: 'kibana_started.elasticsearch.waitTime',
          category: 'database',
          duration: elasticsearchWaitTime,
          type: 'connection'
        }
      });

      // Ensure inline scripting is enabled on the ES cluster
      const scriptingEnabled = await (0, _is_scripting_enabled.isInlineScriptingEnabled)({
        client: this.client.asInternalUser
      });
      if (!scriptingEnabled) {
        throw new Error('Inline scripting is disabled on the Elasticsearch cluster, and is mandatory for Kibana to function. ' + 'Please enabled inline scripting, then restart Kibana. ' + 'Refer to https://www.elastic.co/guide/en/elasticsearch/reference/master/modules-scripting-security.html for more info.');
      }
      capabilities = (0, _get_capabilities.getElasticsearchCapabilities)({
        clusterInfo: await (0, _rxjs.firstValueFrom)(this.clusterInfo$)
      });
    } else {
      // skipStartupConnectionCheck is only used for unit testing, we default to base capabilities
      capabilities = {
        serverless: false
      };
      elasticsearchWaitTime = 0;
    }
    return {
      client: {
        asInternalUser: this.client.asInternalUser,
        asScoped: this.client.asScoped.bind(this.client)
      },
      createClient: (type, clientConfig) => this.createClusterClient(type, config, clientConfig),
      getCapabilities: () => capabilities,
      metrics: {
        elasticsearchWaitTime
      },
      publicBaseUrl: config.publicBaseUrl
    };
  }
  async stop() {
    this.log.debug('Stopping elasticsearch service');
    this.stop$.next();
    if (this.client) {
      await this.client.close();
    }
  }
  createClusterClient(type, baseConfig, clientConfig = {}) {
    const config = (0, _merge_config.mergeConfig)(baseConfig, clientConfig);
    return new _coreElasticsearchClientServerInternal.ClusterClient({
      config,
      logger: this.coreContext.logger.get('elasticsearch'),
      type,
      authHeaders: this.authHeaders,
      getExecutionContext: () => {
        var _this$executionContex;
        return (_this$executionContex = this.executionContextClient) === null || _this$executionContex === void 0 ? void 0 : _this$executionContex.getAsHeader();
      },
      getUnauthorizedErrorHandler: () => this.unauthorizedErrorHandler,
      agentFactoryProvider: this.getAgentManager(baseConfig),
      kibanaVersion: this.kibanaVersion
    });
  }
  getAgentManager({
    dnsCacheTtl
  }) {
    if (!this.agentManager) {
      var _dnsCacheTtl$asSecond;
      this.agentManager = new _coreElasticsearchClientServerInternal.AgentManager(this.log.get('agent-manager'), {
        dnsCacheTtlInSeconds: (_dnsCacheTtl$asSecond = dnsCacheTtl === null || dnsCacheTtl === void 0 ? void 0 : dnsCacheTtl.asSeconds()) !== null && _dnsCacheTtl$asSecond !== void 0 ? _dnsCacheTtl$asSecond : 0 // it should always exists, but some test shortcuts and mocks break this assumption
      });
    }
    return this.agentManager;
  }
}
exports.ElasticsearchService = ElasticsearchService;