"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSpanLinksDetails = getSpanLinksDetails;
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _lodash = require("lodash");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _utils2 = require("./utils");
var _parse_otel_duration = require("../../lib/helpers/parse_otel_duration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchSpanLinksDetails({
  apmEventClient,
  kuery,
  spanLinks,
  start,
  end
}) {
  const {
    startWithBuffer,
    endWithBuffer
  } = (0, _utils2.getBufferedTimerange)({
    start,
    end
  });
  const requiredFields = (0, _as_mutable_array.asMutableArray)([_apm.TRACE_ID, _apm.SERVICE_NAME]);
  const requiredTxFields = (0, _as_mutable_array.asMutableArray)([_apm.TRANSACTION_ID, _apm.TRANSACTION_NAME, _apm.TRANSACTION_DURATION, _apm.AGENT_NAME]);
  const requiredSpanFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.SPAN_NAME, _apm.SPAN_DURATION, _apm.SPAN_SUBTYPE, _apm.SPAN_TYPE, _apm.AGENT_NAME]);
  const requiredUnprocessedOtelSpanFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.SPAN_NAME, _apm.DURATION, _apm.KIND, _apm.SERVICE_LANGUAGE_NAME]);
  const optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.SERVICE_ENVIRONMENT, _apm.PROCESSOR_EVENT]);
  const response = await apmEventClient.search('get_span_links_details', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    fields: [...requiredFields, ...requiredTxFields, ...requiredSpanFields, ...requiredUnprocessedOtelSpanFields, ...optionalFields],
    track_total_hits: false,
    size: 1000,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(startWithBuffer, endWithBuffer), ...(0, _server.kqlQuery)(kuery), {
          bool: {
            should: spanLinks.map(item => {
              return {
                bool: {
                  filter: [{
                    term: {
                      [_apm.TRACE_ID]: item.trace.id
                    }
                  }, {
                    bool: {
                      should: [{
                        term: {
                          [_apm.SPAN_ID]: item.span.id
                        }
                      }, {
                        term: {
                          [_apm.TRANSACTION_ID]: item.span.id
                        }
                      }],
                      minimum_should_match: 1
                    }
                  }]
                }
              };
            }),
            minimum_should_match: 1
          }
        }]
      }
    }
  }, {
    skipProcessorEventFilter: true
  });
  const spanIdsMap = (0, _lodash.keyBy)(spanLinks, 'span.id');
  return response.hits.hits.filter(hit => {
    var _hit$fields$PROCESSOR;
    // The above query might return other spans from the same transaction because siblings spans share the same transaction.id
    // so, if it is a span we need to guarantee that the span.id is the same as the span links ids
    if (((_hit$fields$PROCESSOR = hit.fields[_apm.PROCESSOR_EVENT]) === null || _hit$fields$PROCESSOR === void 0 ? void 0 : _hit$fields$PROCESSOR[0]) === _common.ProcessorEvent.span) {
      const spanLink = (0, _utils.unflattenKnownApmEventFields)(hit.fields, [...requiredFields, ...requiredSpanFields]);
      const hasSpanId = Boolean(spanIdsMap[spanLink.span.id] || false);
      return hasSpanId;
    }
    return true;
  }).map(hit => {
    var _commonEvent$processo;
    const commonEvent = (0, _utils.unflattenKnownApmEventFields)(hit.fields, requiredFields);
    if ((_commonEvent$processo = commonEvent.processor) !== null && _commonEvent$processo !== void 0 && _commonEvent$processo.event) {
      var _commonEvent$transact;
      const commonDetails = {
        serviceName: commonEvent.service.name,
        environment: commonEvent.service.environment,
        transactionId: (_commonEvent$transact = commonEvent.transaction) === null || _commonEvent$transact === void 0 ? void 0 : _commonEvent$transact.id
      };
      if (commonEvent.processor.event === _common.ProcessorEvent.transaction) {
        const event = (0, _utils.unflattenKnownApmEventFields)(hit.fields, [...requiredFields, ...requiredTxFields]);
        return {
          traceId: event.trace.id,
          spanId: event.transaction.id,
          transactionId: event.transaction.id,
          details: {
            ...commonDetails,
            agentName: event.agent.name,
            spanName: event.transaction.name,
            duration: event.transaction.duration.us
          }
        };
      } else {
        const event = (0, _utils.unflattenKnownApmEventFields)(hit.fields, [...requiredFields, ...requiredSpanFields]);
        return {
          traceId: event.trace.id,
          spanId: event.span.id,
          details: {
            ...commonDetails,
            agentName: event.agent.name,
            spanName: event.span.name,
            duration: event.span.duration.us,
            spanSubtype: event.span.subtype,
            spanType: event.span.type
          }
        };
      }
    }
    const event = (0, _utils.unflattenKnownApmEventFields)(hit.fields, [...requiredFields, ...requiredUnprocessedOtelSpanFields]);
    return {
      traceId: event.trace.id,
      spanId: event.span.id,
      details: {
        serviceName: commonEvent.service.name,
        agentName: event.service.language.name,
        spanName: event.span.name,
        duration: (0, _parse_otel_duration.parseOtelDuration)(event.duration),
        environment: event.service.environment
      }
    };
  });
}
async function getSpanLinksDetails({
  apmEventClient,
  spanLinks,
  kuery,
  start,
  end
}) {
  if (!spanLinks.length) {
    return [];
  }

  // chunk span links to avoid too_many_nested_clauses problem
  const spanLinksChunks = (0, _lodash.chunk)(spanLinks, 500);
  const chunkedResponses = await Promise.all(spanLinksChunks.map(spanLinksChunk => fetchSpanLinksDetails({
    apmEventClient,
    kuery,
    spanLinks: spanLinksChunk,
    start,
    end
  })));
  const linkedSpans = chunkedResponses.flat();

  // Creates a map for all span links details found
  const spanLinksDetailsMap = linkedSpans.reduce((acc, spanLink) => {
    if (spanLink.transactionId) {
      const key = `${spanLink.traceId}:${spanLink.transactionId}`;
      acc[key] = {
        traceId: spanLink.traceId,
        spanId: spanLink.transactionId,
        details: spanLink.details
      };
    } else {
      const key = `${spanLink.traceId}:${spanLink.spanId}`;
      acc[key] = {
        traceId: spanLink.traceId,
        spanId: spanLink.spanId,
        details: spanLink.details
      };
    }
    return acc;
  }, {});

  // It's important to keep the original order of the span links,
  // so loops trough the original list merging external links and links with details.
  // external links are links that the details were not found in the ES query.
  return (0, _lodash.compact)(spanLinks.map(item => {
    const key = `${item.trace.id}:${item.span.id}`;
    const details = spanLinksDetailsMap[key];
    if (details) {
      return details;
    }

    // When kuery is not set, returns external links, if not hides this item.
    return (0, _lodash.isEmpty)(kuery) ? {
      traceId: item.trace.id,
      spanId: item.span.id
    } : undefined;
  }));
}