"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateGaps = void 0;
var _lodash = require("lodash");
var _alerting_event_logger = require("../../alerting_event_logger/alerting_event_logger");
var _process_all_rule_gaps = require("../process_all_rule_gaps");
var _update_gaps_batch = require("./update_gaps_batch");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Update gaps for a given rule
 * Prepare gaps for update
 * Update them in bulk
 * If there are conflicts, retry the failed gaps
 * If gaps are passed in, it skips fetching and process them instead
 */
const updateGaps = async params => {
  const {
    ruleId,
    start,
    end,
    logger,
    eventLogClient,
    eventLogger,
    backfillSchedule,
    savedObjectsRepository,
    shouldRefetchAllBackfills,
    backfillClient,
    actionsClient,
    gaps
  } = params;
  if (!eventLogger) {
    throw new Error('Event logger is required');
  }
  try {
    const alertingEventLogger = new _alerting_event_logger.AlertingEventLogger(eventLogger);
    let hasErrors = false;
    const processGapsBatch = async fetchedGaps => {
      if (fetchedGaps.length > 0) {
        const success = await (0, _update_gaps_batch.updateGapsBatch)({
          gaps: fetchedGaps,
          backfillSchedule,
          savedObjectsRepository,
          shouldRefetchAllBackfills,
          backfillClient,
          actionsClient,
          alertingEventLogger,
          logger,
          ruleId,
          eventLogClient
        });
        if (!success) {
          hasErrors = true;
        }
      }
      // Return an object indicating how many gaps were processed per rule id
      return Object.entries((0, _lodash.groupBy)(fetchedGaps, 'ruleId')).reduce((acc, [currentRuleId, currentRuleGaps]) => {
        acc[currentRuleId] = currentRuleGaps.length;
        return acc;
      }, {});
    };
    if (gaps) {
      // If the list of gaps were passed into the function, proceed to update them
      for (const gapsChunk of (0, _lodash.chunk)(gaps, _process_all_rule_gaps.PROCESS_GAPS_DEFAULT_PAGE_SIZE)) {
        await processGapsBatch(gapsChunk);
      }
    } else {
      // Otherwise fetch and update them
      await (0, _process_all_rule_gaps.processAllRuleGaps)({
        ruleIds: [ruleId],
        start: start.toISOString(),
        end: end.toISOString(),
        logger,
        eventLogClient,
        processGapsBatch
      });
    }
    if (hasErrors) {
      throw new Error('Some gaps failed to update');
    }
  } catch (e) {
    logger.error(`Failed to update gaps for rule ${ruleId} from: ${start.toISOString()} to: ${end.toISOString()}: ${e.message}`);
  }
};
exports.updateGaps = updateGaps;