"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AlertingEventLogger = void 0;
exports.createActionExecuteRecord = createActionExecuteRecord;
exports.createAlertRecord = createAlertRecord;
exports.createExecuteTimeoutRecord = createExecuteTimeoutRecord;
exports.createGapRecord = createGapRecord;
exports.executionType = void 0;
exports.initializeExecuteBackfillRecord = initializeExecuteBackfillRecord;
exports.initializeExecuteRecord = initializeExecuteRecord;
exports.updateEvent = updateEvent;
exports.updateEventWithRuleData = updateEventWithRuleData;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var uuid = _interopRequireWildcard(require("uuid"));
var _server = require("@kbn/event-log-plugin/server");
var _plugin = require("../../plugin");
var _saved_objects = require("../../saved_objects");
var _create_alert_event_log_record_object = require("../create_alert_event_log_record_object");
var _gap = require("../rule_gaps/gap");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// 1,000,000 nanoseconds in 1 millisecond
const Millis2Nanos = 1000 * 1000;
const executionType = exports.executionType = {
  STANDARD: 'standard',
  BACKFILL: 'backfill'
};
class AlertingEventLogger {
  constructor(eventLogger) {
    (0, _defineProperty2.default)(this, "eventLogger", void 0);
    (0, _defineProperty2.default)(this, "isInitialized", false);
    (0, _defineProperty2.default)(this, "startTime", void 0);
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "ruleData", void 0);
    (0, _defineProperty2.default)(this, "relatedSavedObjects", []);
    (0, _defineProperty2.default)(this, "executionType", executionType.STANDARD);
    // this is the "execute" event that will be updated over the lifecycle of this class
    (0, _defineProperty2.default)(this, "event", void 0);
    this.eventLogger = eventLogger;
  }

  // For testing purposes
  getEvent() {
    return this.event;
  }
  initialize({
    context,
    runDate,
    ruleData,
    type = executionType.STANDARD
  }) {
    if (this.isInitialized || !context) {
      throw new Error('AlertingEventLogger already initialized');
    }
    this.context = context;
    this.ruleData = ruleData;
    this.executionType = type;
    this.startTime = runDate;
    const ctx = {
      ...this.context,
      taskScheduleDelay: this.startTime.getTime() - this.context.taskScheduledAt.getTime()
    };

    // Populate the "execute" event based on execution type
    switch (type) {
      case executionType.BACKFILL:
        this.initializeBackfill(ctx);
        break;
      default:
        this.initializeStandard(ctx, ruleData);
    }
    this.isInitialized = true;
    this.eventLogger.startTiming(this.event, this.startTime);
  }
  initializeBackfill(ctx) {
    this.relatedSavedObjects = [{
      id: ctx.savedObjectId,
      type: ctx.savedObjectType,
      namespace: ctx.namespace,
      relation: _server.SAVED_OBJECT_REL_PRIMARY
    }];

    // not logging an execute-start event for backfills so just fill in the initial event
    this.event = initializeExecuteBackfillRecord(ctx, this.relatedSavedObjects);
  }
  initializeStandard(ctx, ruleData) {
    if (!ruleData) {
      throw new Error('AlertingEventLogger requires rule data');
    }
    this.relatedSavedObjects = [{
      id: ctx.savedObjectId,
      type: ctx.savedObjectType,
      typeId: ruleData.type.id,
      namespace: ctx.namespace,
      relation: _server.SAVED_OBJECT_REL_PRIMARY
    }];

    // Initialize the "execute" event
    this.event = initializeExecuteRecord(ctx, ruleData, this.relatedSavedObjects);

    // Create and log "execute-start" event
    const executeStartEvent = {
      ...this.event,
      event: {
        ...this.event.event,
        action: _plugin.EVENT_LOG_ACTIONS.executeStart,
        ...(this.startTime ? {
          start: this.startTime.toISOString()
        } : {})
      },
      message: `rule execution start: "${ruleData.id}"`
    };
    this.logEventWithFixedUuid(executeStartEvent);
  }
  getStartAndDuration() {
    return {
      start: this.startTime,
      duration: this.startTime ? (0, _server.millisToNanos)(new Date().getTime() - this.startTime.getTime()) : '0'
    };
  }
  addOrUpdateRuleData({
    name,
    id,
    consumer,
    type,
    revision
  }) {
    if (!this.isInitialized) {
      throw new Error(`AlertingEventLogger not initialized`);
    }
    if (!this.ruleData) {
      if (!id || !type) throw new Error(`Cannot update rule data before it is initialized`);
      this.ruleData = {
        id,
        type
      };
    }
    if (name) {
      this.ruleData.name = name;
    }
    if (consumer) {
      this.ruleData.consumer = consumer;
    }
    if (revision) {
      this.ruleData.revision = revision;
    }
    let updatedRelatedSavedObjects = false;
    if (id && type) {
      // add this to saved objects array if it doesn't already exists
      if (!this.relatedSavedObjects.find(so => so.id === id && so.typeId === type.id)) {
        var _this$context;
        updatedRelatedSavedObjects = true;
        this.relatedSavedObjects.push({
          id: id,
          typeId: type === null || type === void 0 ? void 0 : type.id,
          type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
          namespace: (_this$context = this.context) === null || _this$context === void 0 ? void 0 : _this$context.namespace,
          relation: _server.SAVED_OBJECT_REL_PRIMARY
        });
      }
    }
    updateEventWithRuleData(this.event, {
      ruleName: name,
      ruleId: id,
      ruleType: type,
      consumer,
      revision,
      savedObjects: updatedRelatedSavedObjects ? this.relatedSavedObjects : undefined
    });
  }
  setExecutionSucceeded(message) {
    if (!this.isInitialized || !this.event) {
      throw new Error('AlertingEventLogger not initialized');
    }
    updateEvent(this.event, {
      message,
      outcome: 'success',
      alertingOutcome: 'success'
    });
  }
  setMaintenanceWindowIds(maintenanceWindowIds) {
    if (!this.isInitialized || !this.event) {
      throw new Error('AlertingEventLogger not initialized');
    }
    updateEvent(this.event, {
      maintenanceWindowIds
    });
  }
  setExecutionFailed(message, errorMessage) {
    if (!this.isInitialized || !this.event) {
      throw new Error('AlertingEventLogger not initialized');
    }
    updateEvent(this.event, {
      message,
      outcome: 'failure',
      alertingOutcome: 'failure',
      error: errorMessage
    });
  }
  logTimeout({
    backfill
  } = {}) {
    if (!this.isInitialized || !this.context) {
      throw new Error('AlertingEventLogger not initialized');
    }
    if (backfill && this.executionType !== executionType.BACKFILL) {
      throw new Error('Cannot set backfill fields for non-backfill event log doc');
    }
    const executeTimeoutEvent = createExecuteTimeoutRecord(this.context, this.relatedSavedObjects, this.executionType, this.ruleData);
    if (backfill) {
      updateEvent(executeTimeoutEvent, {
        backfill
      });
    }
    this.logEventWithFixedUuid(executeTimeoutEvent);
  }
  logAlert(alert) {
    if (!this.isInitialized || !this.context || !this.ruleData) {
      throw new Error('AlertingEventLogger not initialized');
    }
    this.logEventWithFixedUuid(createAlertRecord(this.context, this.ruleData, this.relatedSavedObjects, alert));
  }
  logAction(action) {
    if (!this.isInitialized || !this.context || !this.ruleData) {
      throw new Error('AlertingEventLogger not initialized');
    }
    this.logEventWithFixedUuid(createActionExecuteRecord(this.context, this.ruleData, this.relatedSavedObjects, action));
  }
  done({
    status,
    metrics,
    timings,
    backfill
  }) {
    if (!this.isInitialized || !this.event || !this.context) {
      throw new Error('AlertingEventLogger not initialized');
    }
    if (backfill && this.executionType !== executionType.BACKFILL) {
      throw new Error('Cannot set backfill fields for non-backfill event log doc');
    }
    this.eventLogger.stopTiming(this.event);
    if (status) {
      updateEvent(this.event, {
        status: status.status
      });
      if (status.error) {
        var _this$ruleData$type, _status$error, _this$event, _this$event$error, _this$event$event;
        const message = this.ruleData ? `${(_this$ruleData$type = this.ruleData.type) === null || _this$ruleData$type === void 0 ? void 0 : _this$ruleData$type.id}:${this.context.savedObjectId}: execution failed` : `${this.context.savedObjectId}: execution failed`;
        updateEvent(this.event, {
          outcome: 'failure',
          alertingOutcome: 'failure',
          reason: ((_status$error = status.error) === null || _status$error === void 0 ? void 0 : _status$error.reason) || 'unknown',
          error: ((_this$event = this.event) === null || _this$event === void 0 ? void 0 : (_this$event$error = _this$event.error) === null || _this$event$error === void 0 ? void 0 : _this$event$error.message) || status.error.message,
          ...(this.event.message && ((_this$event$event = this.event.event) === null || _this$event$event === void 0 ? void 0 : _this$event$event.outcome) === 'failure' ? {} : {
            message
          })
        });
      } else {
        if (status.warning) {
          var _status$warning, _status$warning2, _this$event2;
          updateEvent(this.event, {
            alertingOutcome: 'warning',
            reason: ((_status$warning = status.warning) === null || _status$warning === void 0 ? void 0 : _status$warning.reason) || 'unknown',
            message: ((_status$warning2 = status.warning) === null || _status$warning2 === void 0 ? void 0 : _status$warning2.message) || ((_this$event2 = this.event) === null || _this$event2 === void 0 ? void 0 : _this$event2.message)
          });
        }
      }
    }
    if (metrics) {
      updateEvent(this.event, {
        metrics
      });
    }
    if (timings) {
      updateEvent(this.event, {
        timings
      });
    }
    if (backfill) {
      updateEvent(this.event, {
        backfill
      });
    }
    this.logEventWithFixedUuid(this.event);
  }
  reportGap({
    gap
  }) {
    if (!this.isInitialized || !this.context || !this.ruleData) {
      throw new Error('AlertingEventLogger not initialized');
    }
    const gapToReport = new _gap.Gap({
      ruleId: this.ruleData.id,
      range: gap
    });
    this.logEventWithFixedUuid(createGapRecord(this.context, this.ruleData, this.relatedSavedObjects, gapToReport.toObject()));
  }
  async updateGaps(docs) {
    return this.eventLogger.updateEvents(docs.map(doc => ({
      event: {
        kibana: {
          alert: {
            rule: {
              gap: doc.gap
            }
          }
        }
      },
      internalFields: doc.internalFields
    })));
  }
  logEventWithFixedUuid(event) {
    this.eventLogger.logEvent(event, uuid.v4());
  }
}
exports.AlertingEventLogger = AlertingEventLogger;
function createAlertRecord(context, ruleData, savedObjects, alert) {
  return (0, _create_alert_event_log_record_object.createAlertEventLogRecordObject)({
    ruleId: ruleData.id,
    ruleType: ruleData.type,
    consumer: ruleData.consumer,
    namespace: context.namespace,
    spaceId: context.spaceId,
    executionId: context.executionId,
    alertUuid: alert.uuid,
    action: alert.action,
    state: alert.state,
    instanceId: alert.id,
    group: alert.group,
    message: alert.message,
    savedObjects,
    ruleName: ruleData.name,
    flapping: alert.flapping,
    maintenanceWindowIds: alert.maintenanceWindowIds,
    ruleRevision: ruleData.revision
  });
}
function createActionExecuteRecord(context, ruleData, savedObjects, action) {
  var _ruleData$type;
  return (0, _create_alert_event_log_record_object.createAlertEventLogRecordObject)({
    ruleId: ruleData.id,
    ruleType: ruleData.type,
    consumer: ruleData.consumer,
    namespace: context.namespace,
    spaceId: context.spaceId,
    executionId: context.executionId,
    action: _plugin.EVENT_LOG_ACTIONS.executeAction,
    instanceId: action.alertId,
    group: action.alertGroup,
    message: `alert: ${(_ruleData$type = ruleData.type) === null || _ruleData$type === void 0 ? void 0 : _ruleData$type.id}:${ruleData.id}: '${ruleData.name}' instanceId: '${action.alertId}' scheduled actionGroup: '${action.alertGroup}' action: ${action.typeId}:${action.id}`,
    savedObjects: [...savedObjects, {
      type: 'action',
      id: action.id,
      typeId: action.typeId
    }],
    ruleName: ruleData.name,
    alertSummary: action.alertSummary,
    ruleRevision: ruleData.revision
  });
}
function createExecuteTimeoutRecord(context, savedObjects, type, ruleData) {
  var _ruleData$type2, _ruleData$name, _ruleData$type3;
  let message = '';
  switch (type) {
    case executionType.BACKFILL:
      message = `backfill "${context.savedObjectId}" cancelled due to timeout`;
      break;
    default:
      message = `rule: ${ruleData === null || ruleData === void 0 ? void 0 : (_ruleData$type2 = ruleData.type) === null || _ruleData$type2 === void 0 ? void 0 : _ruleData$type2.id}:${context.savedObjectId}: '${(_ruleData$name = ruleData === null || ruleData === void 0 ? void 0 : ruleData.name) !== null && _ruleData$name !== void 0 ? _ruleData$name : ''}' execution cancelled due to timeout - exceeded rule type timeout of ${ruleData === null || ruleData === void 0 ? void 0 : (_ruleData$type3 = ruleData.type) === null || _ruleData$type3 === void 0 ? void 0 : _ruleData$type3.ruleTaskTimeout}`;
  }
  return (0, _create_alert_event_log_record_object.createAlertEventLogRecordObject)({
    ruleId: ruleData === null || ruleData === void 0 ? void 0 : ruleData.id,
    ruleType: ruleData === null || ruleData === void 0 ? void 0 : ruleData.type,
    consumer: ruleData === null || ruleData === void 0 ? void 0 : ruleData.consumer,
    namespace: context.namespace,
    spaceId: context.spaceId,
    executionId: context.executionId,
    action: _plugin.EVENT_LOG_ACTIONS.executeTimeout,
    message,
    savedObjects,
    ruleName: ruleData === null || ruleData === void 0 ? void 0 : ruleData.name,
    ruleRevision: ruleData === null || ruleData === void 0 ? void 0 : ruleData.revision
  });
}
function createGapRecord(context, ruleData, savedObjects, gap) {
  return (0, _create_alert_event_log_record_object.createAlertEventLogRecordObject)({
    ruleId: ruleData === null || ruleData === void 0 ? void 0 : ruleData.id,
    ruleType: ruleData === null || ruleData === void 0 ? void 0 : ruleData.type,
    consumer: ruleData === null || ruleData === void 0 ? void 0 : ruleData.consumer,
    namespace: context.namespace,
    spaceId: context.spaceId,
    executionId: context.executionId,
    action: _plugin.EVENT_LOG_ACTIONS.gap,
    savedObjects,
    ruleName: ruleData === null || ruleData === void 0 ? void 0 : ruleData.name,
    ruleRevision: ruleData === null || ruleData === void 0 ? void 0 : ruleData.revision,
    gap
  });
}
function initializeExecuteRecord(context, ruleData, so) {
  return (0, _create_alert_event_log_record_object.createAlertEventLogRecordObject)({
    ruleId: ruleData.id,
    ruleType: ruleData.type,
    consumer: ruleData.consumer,
    ruleRevision: ruleData.revision,
    namespace: context.namespace,
    spaceId: context.spaceId,
    executionId: context.executionId,
    action: _plugin.EVENT_LOG_ACTIONS.execute,
    task: {
      scheduled: context.taskScheduledAt.toISOString(),
      scheduleDelay: Millis2Nanos * context.taskScheduleDelay
    },
    savedObjects: so
  });
}
function initializeExecuteBackfillRecord(context, so) {
  return (0, _create_alert_event_log_record_object.createAlertEventLogRecordObject)({
    namespace: context.namespace,
    spaceId: context.spaceId,
    executionId: context.executionId,
    action: _plugin.EVENT_LOG_ACTIONS.executeBackfill,
    task: {
      scheduled: context.taskScheduledAt.toISOString(),
      scheduleDelay: Millis2Nanos * context.taskScheduleDelay
    },
    savedObjects: so
  });
}
function updateEventWithRuleData(event, opts) {
  const {
    ruleName,
    ruleId,
    consumer,
    ruleType,
    revision,
    savedObjects
  } = opts;
  if (!event) {
    throw new Error('Cannot update event because it is not initialized.');
  }
  if (ruleName) {
    event.rule = {
      ...event.rule,
      name: ruleName
    };
  }
  if (ruleId) {
    event.rule = {
      ...event.rule,
      id: ruleId
    };
  }
  if (consumer) {
    event.kibana = event.kibana || {};
    event.kibana.alert = event.kibana.alert || {};
    event.kibana.alert.rule = event.kibana.alert.rule || {};
    event.kibana.alert.rule.consumer = consumer;
  }
  if (ruleType) {
    event.kibana = event.kibana || {};
    event.kibana.alert = event.kibana.alert || {};
    event.kibana.alert.rule = event.kibana.alert.rule || {};
    if (ruleType.id) {
      event.kibana.alert.rule.rule_type_id = ruleType.id;
      event.rule = {
        ...event.rule,
        category: ruleType.id
      };
    }
    if (ruleType.minimumLicenseRequired) {
      event.rule = {
        ...event.rule,
        license: ruleType.minimumLicenseRequired
      };
    }
    if (ruleType.producer) {
      event.rule = {
        ...event.rule,
        ruleset: ruleType.producer
      };
    }
  }
  if (revision) {
    event.kibana = event.kibana || {};
    event.kibana.alert = event.kibana.alert || {};
    event.kibana.alert.rule = event.kibana.alert.rule || {};
    event.kibana.alert.rule.revision = revision;
  }
  if (savedObjects && savedObjects.length > 0) {
    event.kibana = event.kibana || {};
    event.kibana.saved_objects = savedObjects.map(so => ({
      ...(so.relation ? {
        rel: so.relation
      } : {}),
      type: so.type,
      id: so.id,
      type_id: so.typeId,
      namespace: so.namespace
    }));
  }
}
function updateEvent(event, opts) {
  const {
    message,
    outcome,
    error,
    status,
    reason,
    metrics,
    timings,
    alertingOutcome,
    backfill,
    maintenanceWindowIds
  } = opts;
  if (!event) {
    throw new Error('Cannot update event because it is not initialized.');
  }
  if (message) {
    event.message = message;
  }
  if (outcome) {
    event.event = event.event || {};
    event.event.outcome = outcome;
  }
  if (alertingOutcome) {
    event.kibana = event.kibana || {};
    event.kibana.alerting = event.kibana.alerting || {};
    event.kibana.alerting.outcome = alertingOutcome;
  }
  if (error) {
    event.error = event.error || {};
    event.error.message = error;
  }
  if (status) {
    event.kibana = event.kibana || {};
    event.kibana.alerting = event.kibana.alerting || {};
    event.kibana.alerting.status = status;
  }
  if (reason) {
    event.event = event.event || {};
    event.event.reason = reason;
  }
  if (metrics) {
    event.kibana = event.kibana || {};
    event.kibana.alert = event.kibana.alert || {};
    event.kibana.alert.rule = event.kibana.alert.rule || {};
    event.kibana.alert.rule.execution = event.kibana.alert.rule.execution || {};
    event.kibana.alert.rule.execution.metrics = {
      ...event.kibana.alert.rule.execution.metrics,
      number_of_triggered_actions: metrics.numberOfTriggeredActions ? metrics.numberOfTriggeredActions : 0,
      number_of_generated_actions: metrics.numberOfGeneratedActions ? metrics.numberOfGeneratedActions : 0,
      alert_counts: {
        active: metrics.numberOfActiveAlerts ? metrics.numberOfActiveAlerts : 0,
        new: metrics.numberOfNewAlerts ? metrics.numberOfNewAlerts : 0,
        recovered: metrics.numberOfRecoveredAlerts ? metrics.numberOfRecoveredAlerts : 0
      },
      number_of_delayed_alerts: metrics.numberOfDelayedAlerts ? metrics.numberOfDelayedAlerts : 0,
      number_of_searches: metrics.numSearches ? metrics.numSearches : 0,
      es_search_duration_ms: metrics.esSearchDurationMs ? metrics.esSearchDurationMs : 0,
      total_search_duration_ms: metrics.totalSearchDurationMs ? metrics.totalSearchDurationMs : 0
    };
  }
  if (backfill) {
    event.kibana = event.kibana || {};
    event.kibana.alert = event.kibana.alert || {};
    event.kibana.alert.rule = event.kibana.alert.rule || {};
    event.kibana.alert.rule.execution = event.kibana.alert.rule.execution || {};
    event.kibana.alert.rule.execution.backfill = backfill;
  }
  if (timings) {
    event.kibana = event.kibana || {};
    event.kibana.alert = event.kibana.alert || {};
    event.kibana.alert.rule = event.kibana.alert.rule || {};
    event.kibana.alert.rule.execution = event.kibana.alert.rule.execution || {};
    event.kibana.alert.rule.execution.metrics = {
      ...event.kibana.alert.rule.execution.metrics,
      ...timings
    };
  }
  if (maintenanceWindowIds) {
    event.kibana = event.kibana || {};
    event.kibana.alert = event.kibana.alert || {};
    event.kibana.alert.maintenance_window_ids = maintenanceWindowIds;
  }
}