"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCategoryCountRequest = exports.fetchCategoryCounts = void 0;
var _lodash = require("lodash");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _get_category_query = require("@kbn/aiops-log-pattern-analysis/get_category_query");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _constants = require("../constants");
var _get_query_with_params = require("./get_query_with_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getCategoryCountRequest = (params, fieldName, categories, from, to, sampleProbability) => {
  const {
    index
  } = params;
  const query = (0, _get_query_with_params.getQueryWithParams)({
    // This will override the original start/end params if
    // the optional from/to args are provided.
    params: {
      ...params,
      ...(from ? {
        start: from
      } : {}),
      ...(to ? {
        end: to
      } : {})
    }
  });
  const categoryFilters = categories.reduce((acc, category, i) => {
    acc[`category_${i}`] = (0, _get_category_query.getCategoryQuery)(fieldName, [category]);
    return acc;
  }, {});
  const filtersAgg = {
    category_counts: {
      filters: {
        filters: categoryFilters,
        other_bucket: false
      }
    }
  };
  const {
    wrap
  } = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: sampleProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  return {
    index,
    query,
    size: 0,
    track_total_hits: false,
    aggs: wrap(filtersAgg)
  };
};
exports.getCategoryCountRequest = getCategoryCountRequest;
const fetchCategoryCounts = async (esClient, params, fieldName, categories, sampleProbability, from, to, logger, emitError, abortSignal) => {
  const updatedCategories = (0, _lodash.cloneDeep)(categories);
  if (updatedCategories.categories.length === 0) {
    return updatedCategories;
  }
  const {
    unwrap
  } = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: sampleProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  const request = getCategoryCountRequest(params, fieldName, updatedCategories.categories, from, to, sampleProbability);
  let response;
  try {
    response = await esClient.search(request, {
      signal: abortSignal,
      maxRetries: 0
    });
  } catch (error) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(error)) {
      if (logger) {
        logger.error(`Failed to fetch category counts for field name "${fieldName}", got: \n${JSON.stringify(error, null, 2)}`);
      }
      if (emitError) {
        emitError(`Failed to fetch category counts for field name "${fieldName}".`);
      }
    }
    return updatedCategories;
  }
  if (!response.aggregations) {
    logger === null || logger === void 0 ? void 0 : logger.error(`No aggregations in category counts response for field "${fieldName}".`);
    if (emitError) {
      emitError(`Failed to fetch category counts for field name "${fieldName}".`);
    }
    return updatedCategories;
  }
  const {
    category_counts: categoryCounts
  } = unwrap(response.aggregations);
  for (const [i, category] of updatedCategories.categories.entries()) {
    var _categoryCounts$bucke, _categoryCounts$bucke2;
    category.count = (_categoryCounts$bucke = (_categoryCounts$bucke2 = categoryCounts.buckets[`category_${i}`]) === null || _categoryCounts$bucke2 === void 0 ? void 0 : _categoryCounts$bucke2.doc_count) !== null && _categoryCounts$bucke !== void 0 ? _categoryCounts$bucke : 0;
  }
  return updatedCategories;
};
exports.fetchCategoryCounts = fetchCategoryCounts;