"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TimeSeriesQuerySchema = exports.TimeSeriesConditionSchema = void 0;
exports.validateDuration = validateDuration;
var _i18n = require("@kbn/i18n");
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/alerting-plugin/server");
var _index_threshold = require("@kbn/response-ops-rule-params/index_threshold");
var _data = require("../../../common/data");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// The parameters and response for the `timeSeriesQuery()` service function,
// and associated HTTP endpoint.

// The parameters here are very similar to the alert parameters.
// Missing are `comparator` and `threshold`, which aren't needed to generate
// data values, only needed when evaluating the data.
// Additional parameters are used to indicate the date range of the search,
// and the interval.

const TimeSeriesQuerySchema = exports.TimeSeriesQuerySchema = _configSchema.schema.object({
  ..._index_threshold.CoreQueryParamsSchemaProperties,
  // start of the date range to search, as an iso string; defaults to dateEnd
  dateStart: _configSchema.schema.maybe(_configSchema.schema.string({
    validate: validateDate
  })),
  // end of the date range to search, as an iso string; defaults to now
  dateEnd: _configSchema.schema.maybe(_configSchema.schema.string({
    validate: validateDate
  })),
  // intended to be set to the `interval` property of the alert itself,
  // this value indicates the amount of time between time series dates
  // that will be calculated.
  interval: _configSchema.schema.maybe(_configSchema.schema.string({
    validate: validateDuration
  }))
}, {
  validate: validateBody
});
const TimeSeriesConditionSchema = exports.TimeSeriesConditionSchema = _configSchema.schema.object({
  resultLimit: _configSchema.schema.number(),
  conditionScript: _configSchema.schema.string({
    minLength: 1
  })
});
// using direct type not allowed, circular reference, so body is typed to unknown
function validateBody(anyParams) {
  // validate core query parts, return if it fails validation (returning string)
  const coreQueryValidated = (0, _index_threshold.validateCoreQueryBody)(anyParams);
  if (coreQueryValidated) return coreQueryValidated;
  const {
    dateStart,
    dateEnd,
    interval
  } = anyParams;

  // dates already validated in validateDate(), if provided
  const epochStart = dateStart ? Date.parse(dateStart) : undefined;
  const epochEnd = dateEnd ? Date.parse(dateEnd) : undefined;
  if (epochStart && epochEnd) {
    if (epochStart > epochEnd) {
      return (0, _data.getDateStartAfterDateEndErrorMessage)();
    }
    if (epochStart !== epochEnd && !interval) {
      return _i18n.i18n.translate('xpack.triggersActionsUI.data.coreQueryParams.intervalRequiredErrorMessage', {
        defaultMessage: '[interval]: must be specified if [dateStart] does not equal [dateEnd]'
      });
    }
    if (interval) {
      const intervalMillis = (0, _server.parseDuration)(interval);
      const intervals = Math.round((epochEnd - epochStart) / intervalMillis);
      if (intervals > _data.MAX_INTERVALS) {
        return (0, _data.getTooManyIntervalsErrorMessage)(intervals, _data.MAX_INTERVALS);
      }
    }
  }
}
function validateDate(dateString) {
  const parsed = Date.parse(dateString);
  if (isNaN(parsed)) {
    return _i18n.i18n.translate('xpack.triggersActionsUI.data.coreQueryParams.invalidDateErrorMessage', {
      defaultMessage: 'invalid date {date}',
      values: {
        date: dateString
      }
    });
  }
}
function validateDuration(duration) {
  try {
    (0, _server.parseDuration)(duration);
  } catch (err) {
    return _i18n.i18n.translate('xpack.triggersActionsUI.data.coreQueryParams.invalidDurationErrorMessage', {
      defaultMessage: 'invalid duration: "{duration}"',
      values: {
        duration
      }
    });
  }
}