"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CostCapacity = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utilityTypes = require("utility-types");
var _config = require("../config");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CostCapacity {
  constructor(opts) {
    (0, _defineProperty2.default)(this, "maxAllowedCost", _config.DEFAULT_CAPACITY);
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.logger = opts.logger;
    opts.capacity$.subscribe(capacity => {
      // Capacity config describes the number of normal-cost tasks that can be
      // run simulatenously. Multiple by the cost of a normal cost to determine
      // the maximum allowed cost
      this.maxAllowedCost = (0, _utils.getCapacityInCost)(capacity);
      this.logger.debug(`Task pool now using ${this.maxAllowedCost} as the max allowed cost which is based on a capacity of ${capacity}`);
    });
  }
  get capacity() {
    return this.maxAllowedCost;
  }

  /**
   * Gets how much capacity is currently in use.
   */
  usedCapacity(tasksInPool) {
    let result = 0;
    tasksInPool.forEach(task => {
      var _task$definition;
      if ((_task$definition = task.definition) !== null && _task$definition !== void 0 && _task$definition.cost) {
        result += task.definition.cost;
      }
    });
    return result;
  }

  /**
   * Gets % of capacity in use
   */
  usedCapacityPercentage(tasksInPool) {
    return this.capacity ? Math.round(this.usedCapacity(tasksInPool) * 100 / this.capacity) : 100;
  }

  /**
   * Gets how much capacity is currently in use by each type.
   */
  getUsedCapacityByType(tasksInPool, type) {
    return tasksInPool.reduce((count, runningTask) => {
      var _runningTask$definiti;
      return ((_runningTask$definiti = runningTask.definition) === null || _runningTask$definiti === void 0 ? void 0 : _runningTask$definiti.type) === type ? count + runningTask.definition.cost : count;
    }, 0);
  }
  availableCapacity(tasksInPool, taskDefinition) {
    const allAvailableCapacity = this.capacity - this.usedCapacity(tasksInPool);
    if (taskDefinition && !(0, _utilityTypes.isNullish)(taskDefinition.maxConcurrency)) {
      // calculate the max capacity that can be used for this task type based on cost
      const maxCapacityForType = taskDefinition.maxConcurrency * taskDefinition.cost;
      return Math.max(Math.min(allAvailableCapacity, maxCapacityForType - this.getUsedCapacityByType([...tasksInPool.values()], taskDefinition.type)), 0);
    }
    return allAvailableCapacity;
  }
  determineTasksToRunBasedOnCapacity(tasks, availableCapacity) {
    const tasksToRun = [];
    const leftOverTasks = [];
    let capacityAccumulator = 0;
    for (const task of tasks) {
      var _task$definition$cost, _task$definition2;
      const taskCost = (_task$definition$cost = (_task$definition2 = task.definition) === null || _task$definition2 === void 0 ? void 0 : _task$definition2.cost) !== null && _task$definition$cost !== void 0 ? _task$definition$cost : 0;
      if (capacityAccumulator + taskCost <= availableCapacity) {
        tasksToRun.push(task);
        capacityAccumulator += taskCost;
      } else {
        leftOverTasks.push(task);
        // Don't claim further tasks even if lower cost tasks are next.
        // It may be an extra large task and we need to make room for it
        // for the next claiming cycle
        capacityAccumulator = availableCapacity;
      }
    }
    return [tasksToRun, leftOverTasks];
  }
}
exports.CostCapacity = CostCapacity;