"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MonitorConfigRepository = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _with_apm_span = require("@kbn/apm-data-access-plugin/server/utils/with_apm_span");
var _lodash = require("lodash");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _common = require("../routes/common");
var _saved_objects = require("../../common/types/saved_objects");
var _utils = require("../synthetics_service/utils");
var _runtime_types = require("../../common/runtime_types");
var _combine_and_sort_saved_objects = require("./utils/combine_and_sort_saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getSuccessfulResult = results => {
  for (const result of results) {
    if (result.status === 'fulfilled') {
      return result.value;
    }
  }
  const firstError = results.find(r => r.status === 'rejected');
  throw (firstError === null || firstError === void 0 ? void 0 : firstError.reason) || new Error('Unknown error');
};
class MonitorConfigRepository {
  constructor(soClient, encryptedSavedObjectsClient, logger // Replace with appropriate logger type
  ) {
    (0, _defineProperty2.default)(this, "handleLegacyFilter", filter => {
      if (!filter) {
        return filter;
      }
      // Replace syntheticsMonitorAttributes with legacyMonitorAttributes in the filter
      return filter.replace(new RegExp(_saved_objects.syntheticsMonitorAttributes, 'g'), _saved_objects.legacyMonitorAttributes);
    });
    this.soClient = soClient;
    this.encryptedSavedObjectsClient = encryptedSavedObjectsClient;
    this.logger = logger;
  }
  async get(id) {
    // we need to resolve both syntheticsMonitorSavedObjectType and legacySyntheticsMonitorTypeSingle
    const results = await this.soClient.bulkGet([{
      type: _saved_objects.syntheticsMonitorSavedObjectType,
      id
    }, {
      type: _saved_objects.legacySyntheticsMonitorTypeSingle,
      id
    }]);
    const resolved = results.saved_objects.find(obj => obj === null || obj === void 0 ? void 0 : obj.attributes);
    if (!resolved) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(_saved_objects.syntheticsMonitorSavedObjectType, id);
    }
    return resolved;
  }
  async getDecrypted(id, spaceId) {
    const namespace = {
      namespace: spaceId
    };

    // Helper to attempt decryption and catch 404
    const tryGetDecrypted = async soType => {
      return await this.encryptedSavedObjectsClient.getDecryptedAsInternalUser(soType, id, namespace);
    };
    const results = await Promise.allSettled([tryGetDecrypted(_saved_objects.syntheticsMonitorSavedObjectType), tryGetDecrypted(_saved_objects.legacySyntheticsMonitorTypeSingle)]);
    const decryptedMonitor = getSuccessfulResult(results);
    return {
      normalizedMonitor: (0, _utils.normalizeSecrets)(decryptedMonitor),
      decryptedMonitor
    };
  }
  async create({
    id,
    spaceId,
    normalizedMonitor,
    savedObjectType
  }) {
    var _spaces;
    let {
      spaces
    } = normalizedMonitor;
    // Ensure spaceId is included in spaces
    if ((0, _lodash.isEmpty)(spaces)) {
      spaces = [spaceId];
    } else if (!((_spaces = spaces) !== null && _spaces !== void 0 && _spaces.includes(spaceId))) {
      var _spaces2;
      spaces = [...((_spaces2 = spaces) !== null && _spaces2 !== void 0 ? _spaces2 : []), spaceId];
    }
    const opts = {
      id,
      ...(id && {
        overwrite: true
      }),
      ...(!(0, _lodash.isEmpty)(spaces) && {
        initialNamespaces: spaces
      })
    };
    return await this.soClient.create(savedObjectType !== null && savedObjectType !== void 0 ? savedObjectType : _saved_objects.syntheticsMonitorSavedObjectType, (0, _utils.formatSecrets)({
      ...normalizedMonitor,
      [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || id,
      [_runtime_types.ConfigKey.CONFIG_ID]: id,
      revision: 1,
      [_runtime_types.ConfigKey.KIBANA_SPACES]: spaces
    }), opts);
  }
  async createBulk({
    monitors,
    savedObjectType
  }) {
    const newMonitors = monitors.map(({
      id,
      monitor
    }) => {
      const {
        spaces
      } = monitor;
      return {
        id,
        type: savedObjectType !== null && savedObjectType !== void 0 ? savedObjectType : _saved_objects.syntheticsMonitorSavedObjectType,
        attributes: (0, _utils.formatSecrets)({
          ...monitor,
          [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: monitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || id,
          [_runtime_types.ConfigKey.CONFIG_ID]: id,
          revision: 1
        }),
        ...(!(0, _lodash.isEmpty)(spaces) && {
          initialNamespaces: spaces
        })
      };
    });
    const result = await this.soClient.bulkCreate(newMonitors);
    return result.saved_objects;
  }
  async update(id, data, decryptedPreviousMonitor) {
    const soType = decryptedPreviousMonitor.type;
    const prevSpaces = (decryptedPreviousMonitor.namespaces || []).sort();
    const spaces = (data.spaces || []).sort();
    // If the spaces have changed, we need to delete the saved object and recreate it
    if ((0, _lodash.isEqual)(prevSpaces, spaces)) {
      return this.soClient.update(soType, id, data);
    } else {
      await this.soClient.delete(soType, id, {
        force: true
      });
      return await this.soClient.create(_saved_objects.syntheticsMonitorSavedObjectType, data, {
        id,
        ...(!(0, _lodash.isEmpty)(spaces) && {
          initialNamespaces: spaces
        })
      });
    }
  }
  async bulkUpdate({
    monitors,
    namespace
  }) {
    // Split monitors into those needing recreation and those that can be updated
    const toRecreate = [];
    const toUpdate = [];
    for (const monitor of monitors) {
      const {
        attributes,
        id,
        previousMonitor
      } = monitor;
      const prevSpaces = (previousMonitor.namespaces || []).sort();
      const spaces = (attributes.spaces || []).sort();
      if (!(0, _lodash.isEqual)(prevSpaces, spaces) && !(0, _lodash.isEmpty)(spaces)) {
        toRecreate.push({
          id,
          attributes,
          previousMonitor
        });
        continue;
      }
      toUpdate.push({
        type: previousMonitor === null || previousMonitor === void 0 ? void 0 : previousMonitor.type,
        id,
        attributes,
        namespace
      });
    }

    // Bulk delete legacy monitors if spaces changed
    if (toRecreate.length > 0) {
      const deleteObjects = toRecreate.map(({
        id,
        previousMonitor
      }) => ({
        id,
        type: previousMonitor.type
      }));
      await this.soClient.bulkDelete(deleteObjects, {
        force: true
      });
    }

    // Use bulkCreate for recreations
    let recreateResults = [];
    if (toRecreate.length > 0) {
      const bulkCreateObjects = toRecreate.map(({
        id,
        attributes,
        previousMonitor
      }) => ({
        id,
        type: _saved_objects.syntheticsMonitorSavedObjectType,
        attributes,
        ...(!(0, _lodash.isEmpty)(attributes.spaces) && {
          initialNamespaces: attributes.spaces
        })
      }));
      const bulkCreateResult = await this.soClient.bulkCreate(bulkCreateObjects);
      recreateResults = bulkCreateResult.saved_objects;
    }

    // Bulk update the rest
    const bulkUpdateResult = toUpdate.length ? await this.soClient.bulkUpdate(toUpdate) : {
      saved_objects: []
    };

    // Combine results
    return {
      saved_objects: [...bulkUpdateResult.saved_objects, ...recreateResults]
    };
  }
  async find(options, types = _saved_objects.syntheticsMonitorSOTypes, soClient = this.soClient) {
    var _options$perPage, _options$page;
    const perPage = (_options$perPage = options.perPage) !== null && _options$perPage !== void 0 ? _options$perPage : 5000;
    const page = (_options$page = options.page) !== null && _options$page !== void 0 ? _options$page : 1;
    // fetch all possible monitors, sort locally since we can't sort across multiple types yet
    const maximumPageSize = 10_000;
    const promises = types.map(type => {
      const opts = {
        type,
        ...options,
        perPage: maximumPageSize,
        page: 1
      };
      return soClient.find(this.handleLegacyOptions(opts, type));
    });
    const results = await Promise.all(promises);

    // Use util to combine, sort, and slice
    return (0, _combine_and_sort_saved_objects.combineAndSortSavedObjects)(results, options, page, perPage);
  }
  async findDecryptedMonitors({
    spaceId,
    filter
  }) {
    const getDecrypted = async soType => {
      // Handle legacy filter if the type is legacy
      const legacyFilter = soType === _saved_objects.legacySyntheticsMonitorTypeSingle ? this.handleLegacyFilter(filter) : filter;
      const finder = await this.encryptedSavedObjectsClient.createPointInTimeFinderDecryptedAsInternalUser({
        filter: legacyFilter,
        type: soType,
        perPage: 500,
        namespaces: [spaceId]
      });
      const decryptedMonitors = [];
      for await (const result of finder.find()) {
        decryptedMonitors.push(...result.saved_objects);
      }
      finder.close().catch(() => {});
      return decryptedMonitors;
    };
    const [decryptedMonitors, legacyDecryptedMonitors] = await Promise.all([getDecrypted(_saved_objects.syntheticsMonitorSavedObjectType), getDecrypted(_saved_objects.legacySyntheticsMonitorTypeSingle)]);
    return [...decryptedMonitors, ...legacyDecryptedMonitors];
  }
  async bulkDelete(monitors) {
    return this.soClient.bulkDelete(monitors, {
      force: true
    });
  }
  async getAll({
    search,
    fields,
    filter,
    sortField = 'name.keyword',
    sortOrder = 'asc',
    searchFields = _common.MONITOR_SEARCH_FIELDS,
    showFromAllSpaces
  }) {
    const getConfigs = async syntheticsMonitorType => {
      const findFilter = syntheticsMonitorType === _saved_objects.legacySyntheticsMonitorTypeSingle ? this.handleLegacyFilter(filter) : filter;
      const finder = this.soClient.createPointInTimeFinder({
        type: syntheticsMonitorType,
        perPage: 5000,
        search,
        sortField,
        sortOrder,
        fields,
        filter: findFilter,
        searchFields,
        ...(showFromAllSpaces && {
          namespaces: ['*']
        })
      });
      const hits = [];
      for await (const result of finder.find()) {
        hits.push(...result.saved_objects);
      }
      finder.close().catch(() => {});
      return hits;
    };
    return (0, _with_apm_span.withApmSpan)('get_all_monitors', async () => {
      const [configs, legacyConfigs] = await Promise.all([getConfigs(_saved_objects.syntheticsMonitorSavedObjectType), getConfigs(_saved_objects.legacySyntheticsMonitorTypeSingle)]);
      return [...configs, ...legacyConfigs];
    });
  }
  handleLegacyOptions(options, type) {
    // convert the options to string and replace if the type is opposite of either of the synthetics monitor types
    try {
      const opts = JSON.stringify(options);
      if (type === _saved_objects.syntheticsMonitorSavedObjectType) {
        return JSON.parse(opts.replace(new RegExp(_saved_objects.legacyMonitorAttributes, 'g'), _saved_objects.syntheticsMonitorAttributes));
      } else if (type === _saved_objects.legacySyntheticsMonitorTypeSingle) {
        return JSON.parse(opts.replace(new RegExp(_saved_objects.syntheticsMonitorAttributes, 'g'), _saved_objects.legacyMonitorAttributes));
      }
    } catch (e) {
      var _this$logger;
      (_this$logger = this.logger) === null || _this$logger === void 0 ? void 0 : _this$logger.error(`Error parsing handleLegacyOptions: ${e}`);
      return options;
    }
  }
}
exports.MonitorConfigRepository = MonitorConfigRepository;