"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncNewMonitorBulk = exports.deleteMonitorIfCreated = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _uuid = require("uuid");
var _monitor_upgrade_sender = require("../../telemetry/monitor_upgrade_sender");
var _runtime_types = require("../../../../common/runtime_types");
var _delete_monitor_api = require("../services/delete_monitor_api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const syncNewMonitorBulk = async ({
  routeContext,
  normalizedMonitors,
  privateLocations,
  spaceId
}) => {
  const {
    server,
    syntheticsMonitorClient,
    monitorConfigRepository,
    request
  } = routeContext;
  const {
    query
  } = request;
  let newMonitors = null;
  const monitorsToCreate = normalizedMonitors.map(monitor => {
    const monitorSavedObjectId = (0, _uuid.v4)();
    return {
      id: monitorSavedObjectId,
      monitor: {
        ...monitor,
        [_runtime_types.ConfigKey.CONFIG_ID]: monitorSavedObjectId,
        [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: monitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || monitorSavedObjectId
      }
    };
  });
  try {
    const [createdMonitors, [policiesResult, syncErrors]] = await Promise.all([monitorConfigRepository.createBulk({
      monitors: monitorsToCreate,
      savedObjectType: query.savedObjectType
    }), syntheticsMonitorClient.addMonitors(monitorsToCreate, privateLocations, spaceId)]);
    let failedMonitors = [];
    const {
      failed: failedPolicies
    } = policiesResult !== null && policiesResult !== void 0 ? policiesResult : {};
    newMonitors = createdMonitors;
    if (failedPolicies && (failedPolicies === null || failedPolicies === void 0 ? void 0 : failedPolicies.length) > 0 && newMonitors) {
      failedMonitors = await handlePrivateConfigErrors(routeContext, newMonitors, failedPolicies);
    }
    sendNewMonitorTelemetry(server, newMonitors, syncErrors);
    return {
      errors: syncErrors,
      newMonitors,
      failedMonitors
    };
  } catch (e) {
    await rollBackNewMonitorBulk(monitorsToCreate, routeContext);
    throw e;
  }
};
exports.syncNewMonitorBulk = syncNewMonitorBulk;
const handlePrivateConfigErrors = async (routeContext, createdMonitors, failedPolicies) => {
  const failedMonitors = [];
  await (0, _pMap.default)(failedPolicies, async ({
    packagePolicy,
    error
  }) => {
    var _enabledInput$streams, _vars$ConfigKey$CONFI;
    const {
      inputs
    } = packagePolicy;
    const enabledInput = inputs === null || inputs === void 0 ? void 0 : inputs.find(input => input.enabled);
    const stream = enabledInput === null || enabledInput === void 0 ? void 0 : (_enabledInput$streams = enabledInput.streams) === null || _enabledInput$streams === void 0 ? void 0 : _enabledInput$streams[0];
    const vars = stream === null || stream === void 0 ? void 0 : stream.vars;
    const monitorId = vars === null || vars === void 0 ? void 0 : (_vars$ConfigKey$CONFI = vars[_runtime_types.ConfigKey.CONFIG_ID]) === null || _vars$ConfigKey$CONFI === void 0 ? void 0 : _vars$ConfigKey$CONFI.value;
    const monitor = createdMonitors.find(savedObject => savedObject.attributes[_runtime_types.ConfigKey.CONFIG_ID] === monitorId);
    if (monitor) {
      failedMonitors.push({
        monitor,
        error
      });
      await deleteMonitorIfCreated({
        routeContext,
        newMonitorId: monitor.id
      });
      createdMonitors.splice(createdMonitors.indexOf(monitor), 1);
    }
  });
  return failedMonitors;
};
const rollBackNewMonitorBulk = async (monitorsToCreate, routeContext) => {
  const {
    server
  } = routeContext;
  try {
    const deleteMonitorAPI = new _delete_monitor_api.DeleteMonitorAPI(routeContext);
    await deleteMonitorAPI.execute({
      monitorIds: monitorsToCreate.map(({
        id
      }) => id)
    });
  } catch (error) {
    // ignore errors here
    server.logger.error(`Unable to rollback new monitors, Error: ${error.message}`, {
      error
    });
  }
};
const sendNewMonitorTelemetry = (server, monitors, errors) => {
  for (const monitor of monitors) {
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryEvent)({
      errors,
      monitor,
      isInlineScript: Boolean(monitor.attributes[_runtime_types.ConfigKey.SOURCE_INLINE]),
      stackVersion: server.stackVersion
    }));
  }
};
const deleteMonitorIfCreated = async ({
  newMonitorId,
  routeContext
}) => {
  const {
    server,
    monitorConfigRepository
  } = routeContext;
  try {
    const encryptedMonitor = await monitorConfigRepository.get(newMonitorId);
    if (encryptedMonitor) {
      const deleteMonitorAPI = new _delete_monitor_api.DeleteMonitorAPI(routeContext);
      await deleteMonitorAPI.deleteMonitorBulk({
        monitors: [encryptedMonitor]
      });
    }
  } catch (e) {
    // ignore errors here
    server.logger.error(`Unable to delete monitor with id ${newMonitorId}`, {
      error: e
    });
  }
};
exports.deleteMonitorIfCreated = deleteMonitorIfCreated;