"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.contentRoutes = void 0;
var _stream = require("stream");
var _zod = require("@kbn/zod");
var _contentPacksSchema = require("@kbn/content-packs-schema");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _constants = require("../../../common/constants");
var _create_server_route = require("../create_server_route");
var _status_error = require("../../lib/streams/errors/status_error");
var _content = require("../../lib/content");
var _stream2 = require("../../lib/content/stream");
var _tree = require("../../lib/content/stream/tree");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CONTENT_PACK_SIZE_BYTES = 1024 * 1024 * 5; // 5MB

const exportContentRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/content/export 2023-10-31',
  options: {
    access: 'public',
    summary: 'Export stream content',
    description: 'Exports the content associated to a stream.'
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      name: _zod.z.string(),
      description: _zod.z.string(),
      version: _zod.z.string(),
      include: _contentPacksSchema.contentPackIncludedObjectsSchema
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  async handler({
    params,
    request,
    response,
    context,
    getScopedClients
  }) {
    await checkEnabled(context);
    const {
      assetClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const root = await streamsClient.getStream(params.path.name);
    if (!_streamsSchema.Streams.WiredStream.Definition.is(root)) {
      throw new _status_error.StatusError('Only wired streams can be exported', 400);
    }
    const [ancestors, descendants] = await Promise.all([streamsClient.getAncestors(params.path.name), streamsClient.getDescendants(params.path.name)]);
    const queryLinks = await assetClient.getAssetLinks([params.path.name, ...descendants.map(stream => stream.name)], ['query']);
    const inheritedFields = (0, _streamsSchema.getInheritedFieldsFromAncestors)(ancestors);
    const exportedTree = (0, _tree.asTree)({
      root: params.path.name,
      include: (0, _stream2.scopeIncludedObjects)({
        root: params.path.name,
        include: params.body.include
      }),
      streams: [root, ...descendants].map(stream => {
        if (stream.name === params.path.name) {
          // merge non-base inherited mappings into the exported root
          stream.ingest.wired.fields = (0, _stream2.withoutBaseFields)({
            ...inheritedFields,
            ...stream.ingest.wired.fields
          });
        }
        return asContentPackEntry({
          stream,
          queryLinks: queryLinks[stream.name]
        });
      })
    });
    const archive = await (0, _content.generateArchive)(params.body, (0, _stream2.prepareStreamsForExport)({
      tree: exportedTree
    }));
    return response.ok({
      body: archive,
      headers: {
        'Content-Disposition': `attachment; filename="${params.body.name}-${params.body.version}.zip"`,
        'Content-Type': 'application/zip'
      }
    });
  }
});
function asContentPackEntry({
  stream,
  queryLinks
}) {
  return {
    type: 'stream',
    name: stream.name,
    request: {
      stream: {
        ...(0, _lodash.omit)(stream, ['name'])
      },
      ..._streamsSchema.emptyAssets,
      queries: queryLinks.map(({
        query
      }) => query)
    }
  };
}
const importContentRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /api/streams/{name}/content/import 2023-10-31',
  options: {
    access: 'public',
    summary: 'Import content into a stream',
    description: 'Links content objects to a stream.',
    body: {
      accepts: 'multipart/form-data',
      maxBytes: MAX_CONTENT_PACK_SIZE_BYTES,
      output: 'stream'
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      include: _zod.z.string().transform(value => _contentPacksSchema.contentPackIncludedObjectsSchema.parse(JSON.parse(value))),
      content: _zod.z.instanceof(_stream.Readable)
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  async handler({
    params,
    request,
    context,
    getScopedClients
  }) {
    await checkEnabled(context);
    const {
      assetClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const root = await streamsClient.getStream(params.path.name);
    if (!_streamsSchema.Streams.WiredStream.Definition.is(root)) {
      throw new _status_error.StatusError('Can only import content into wired streams', 400);
    }
    const contentPack = await (0, _content.parseArchive)(params.body.content);
    const descendants = await streamsClient.getDescendants(params.path.name);
    const queryLinks = await assetClient.getAssetLinks([params.path.name, ...descendants.map(({
      name
    }) => name)], ['query']);
    const existingTree = (0, _tree.asTree)({
      root: params.path.name,
      include: {
        objects: {
          all: {}
        }
      },
      streams: [root, ...descendants].map(stream => asContentPackEntry({
        stream,
        queryLinks: queryLinks[stream.name]
      }))
    });
    const incomingTree = (0, _tree.asTree)({
      root: params.path.name,
      include: (0, _stream2.scopeIncludedObjects)({
        root: params.path.name,
        include: params.body.include
      }),
      streams: (0, _stream2.scopeContentPackStreams)({
        root: params.path.name,
        streams: contentPack.entries.filter(entry => entry.type === 'stream')
      })
    });
    const streams = (0, _stream2.prepareStreamsForImport)({
      existing: existingTree,
      incoming: incomingTree
    });
    return await streamsClient.bulkUpsert(streams);
  }
});
const previewContentRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/content/preview',
  options: {
    access: 'internal',
    summary: 'Preview a content pack',
    description: 'Returns a json representation of a content pack.',
    body: {
      accepts: 'multipart/form-data',
      maxBytes: MAX_CONTENT_PACK_SIZE_BYTES,
      output: 'stream'
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      content: _zod.z.instanceof(_stream.Readable)
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  async handler({
    request,
    params,
    context,
    getScopedClients
  }) {
    await checkEnabled(context);
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    return await (0, _content.parseArchive)(params.body.content);
  }
});
async function checkEnabled(context) {
  const core = await context.core;
  const enabled = await core.uiSettings.client.get(_managementSettingsIds.OBSERVABILITY_STREAMS_ENABLE_CONTENT_PACKS);
  if (!enabled) {
    throw new _status_error.StatusError('Content packs are not enabled', 400);
  }
}
const contentRoutes = exports.contentRoutes = {
  ...exportContentRoute,
  ...importContentRoute,
  ...previewContentRoute
};