"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.State = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _lockManager = require("@kbn/lock-manager");
var _failed_to_apply_requested_changes_error = require("./errors/failed_to_apply_requested_changes_error");
var _failed_to_determine_elasticsearch_actions_error = require("./errors/failed_to_determine_elasticsearch_actions_error");
var _failed_to_load_current_state_error = require("./errors/failed_to_load_current_state_error");
var _failed_to_change_state_error = require("./errors/failed_to_change_state_error");
var _invalid_state_error = require("./errors/invalid_state_error");
var _execution_plan = require("./execution_plan/execution_plan");
var _stream_from_definition = require("./stream_active_record/stream_from_definition");
var _concurrent_access_error = require("./errors/concurrent_access_error");
var _insufficient_permissions_error = require("../errors/insufficient_permissions_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The State class is responsible for moving from the current state to the desired state
 * Based on the requested bulk changes. It follows the following phases to achieve this:
 * 1. Load the current state by reading all the stored Stream definitions
 * 2. Applying the requested changes to a clone of the current state (by showing the change to each Stream instance)
 * 3. Applying cascading changes that the Stream instances return in response to a requested change
 * 4. Validating the desired state by asking each Stream if it is valid in this state
 * 5. If the state is valid, State asks each Stream to determine the required Elasticsearch actions needed to reach the desired state
 * 6. If it is a dry run, it returns the affected streams and the Elasticsearch actions that would have happened
 * 7. If it is a real run, it commits the changes by updating the various Elasticsearch resources (delegated to the ExecutionPlan class)
 * 8. If this fails, it throws an error and guides the user to use resync if needed
 */
class State {
  // Changes to state should only happen via static State.attemptChanges or  State.resync
  // State.currentState can be used to simply read the state
  constructor(streams, dependencies) {
    (0, _defineProperty2.default)(this, "streamsByName", void 0);
    (0, _defineProperty2.default)(this, "dependencies", void 0);
    this.streamsByName = new Map();
    streams.forEach(stream => this.streamsByName.set(stream.definition.name, stream));
    this.dependencies = dependencies;
  }
  clone() {
    const newStreams = this.all().map(stream => stream.clone());
    return new State(newStreams, this.dependencies);
  }
  static async attemptChanges(requestedChanges, dependencies, dryRun = false) {
    const startingState = await State.currentState(dependencies);
    const desiredState = await startingState.applyChanges(requestedChanges);
    await desiredState.validate(startingState);
    if (dryRun) {
      const changes = desiredState.changes(startingState);
      // Do we always want to include/expose the Elasticsearch actions?
      const elasticsearchActions = await desiredState.plannedActions(startingState);
      return {
        status: 'valid_dry_run',
        changes,
        elasticsearchActions
      };
    } else {
      const lmService = dependencies.lockManager;
      return lmService.withLock('streams/apply_changes', async () => {
        try {
          await desiredState.commitChanges(startingState);
          return {
            status: 'success',
            changes: desiredState.changes(startingState)
          };
        } catch (error) {
          var _error$statusCode;
          if (error instanceof _insufficient_permissions_error.InsufficientPermissionsError) {
            throw error;
          }
          throw new _failed_to_change_state_error.FailedToChangeStateError(`Failed to change state: ${error.message}. The stream state may be inconsistent. Revert your last change, or use the resync API to restore a consistent state.`, (_error$statusCode = error.statusCode) !== null && _error$statusCode !== void 0 ? _error$statusCode : 500);
        }
      }).catch(error => {
        if ((0, _lockManager.isLockAcquisitionError)(error)) {
          throw new _concurrent_access_error.ConcurrentAccessError('Could not acquire lock for applying changes');
        }
        throw error;
      });
    }
  }
  static async resync(dependencies) {
    const currentState = await State.currentState(dependencies);

    // This way all current streams will look like they have been added
    currentState.all().map(stream => stream.markAsUpserted());
    const emptyState = new State([], dependencies);

    // We skip validation since we assume the stored state to be correct
    await currentState.commitChanges(emptyState);
  }
  static async currentState(dependencies) {
    try {
      const streamsSearchResponse = await dependencies.storageClient.search({
        size: 10000,
        sort: [{
          name: 'asc'
        }],
        track_total_hits: false
      });
      const streams = streamsSearchResponse.hits.hits.map(({
        _source: definition
      }) => (0, _stream_from_definition.streamFromDefinition)(definition, dependencies));
      return new State(streams, dependencies);
    } catch (error) {
      throw new _failed_to_load_current_state_error.FailedToLoadCurrentStateError(`Failed to load current Streams state: ${error.message}`);
    }
  }
  async applyChanges(requestedChanges) {
    try {
      const desiredState = this.clone();
      let checkingState;
      if (this.dependencies.isDev) {
        checkingState = this.clone();
      }
      for (const requestedChange of requestedChanges) {
        // Apply one change and any cascading changes from that change
        await this.applyRequestedChange(requestedChange, desiredState, this);
      }
      if (this.dependencies.isDev) {
        if (!(0, _lodash.isEqual)(this.toPrintable(), checkingState.toPrintable())) {
          throw new Error('applyChanges resulted in the starting state being modified');
        }
      }
      return desiredState;
    } catch (error) {
      throw new _failed_to_apply_requested_changes_error.FailedToApplyRequestedChangesError(`Failed to apply requested changes to Stream state: ${[error.message]}`, error.statusCode);
    }
  }
  async applyRequestedChange(requestedChange, desiredState, startingState) {
    const cascadingChanges = await this.applyChange(requestedChange, desiredState, startingState);
    await this.applyCascadingChanges(cascadingChanges, desiredState, startingState);
  }
  async applyCascadingChanges(cascadingChanges, desiredState, startingState) {
    let iterationCounter = 0;
    let currentCascadingChanges = [...cascadingChanges];
    while (currentCascadingChanges.length !== 0) {
      const newCascadingChanges = [];
      for (const cascadingChange of currentCascadingChanges) {
        const newChanges = await this.applyChange(cascadingChange, desiredState, startingState);
        newCascadingChanges.push(...newChanges);
      }
      currentCascadingChanges = newCascadingChanges;
      if (++iterationCounter > 100) {
        throw new Error('Excessive cascading changes');
      }
    }
  }
  async applyChange(change, desiredState, startingState) {
    // Add new streams if they haven't already been added by a previous (cascading) change
    if (change.type === 'upsert' && !desiredState.has(change.definition.name)) {
      const newStream = (0, _stream_from_definition.streamFromDefinition)(change.definition, this.dependencies);
      desiredState.set(newStream.definition.name, newStream);
    }
    const cascadingChanges = [];
    for (const stream of desiredState.all()) {
      const newChanges = await stream.applyChange(change, desiredState, startingState);
      cascadingChanges.push(newChanges);
    }
    return cascadingChanges.flat();
  }
  async validate(startingState) {
    const validationResults = await Promise.all(this.all().map(stream => stream.validate(this, startingState)));
    const isValid = validationResults.every(validationResult => validationResult.isValid);
    const errors = validationResults.flatMap(validationResult => validationResult.errors);
    if (!isValid) {
      throw new _invalid_state_error.InvalidStateError(errors, `Desired stream state is invalid`);
    }
  }
  async commitChanges(startingState) {
    const executionPlan = new _execution_plan.ExecutionPlan(this.dependencies);
    await executionPlan.plan(await this.determineElasticsearchActions(this.changedStreams(), this, startingState));
    await executionPlan.execute();
  }
  async determineElasticsearchActions(changedStreams, desiredState, startingState) {
    try {
      const actions = await Promise.all(changedStreams.map(stream => stream.determineElasticsearchActions(desiredState, startingState, startingState.get(stream.definition.name))));
      return actions.flat();
    } catch (error) {
      throw new _failed_to_determine_elasticsearch_actions_error.FailedToDetermineElasticsearchActionsError(`Failed to determine Elasticsearch actions: ${error.message}`);
    }
  }
  changedStreams() {
    return this.all().filter(stream => stream.hasChanged());
  }
  async plannedActions(startingState) {
    const executionPlan = new _execution_plan.ExecutionPlan(this.dependencies);
    await executionPlan.plan(await this.determineElasticsearchActions(this.changedStreams(), this, startingState));
    return executionPlan.plannedActions();
  }
  changes(startingState) {
    const startingStreams = startingState.all().map(stream => stream.definition.name);
    const desiredStreams = this.all().map(stream => stream.definition.name);
    const deleted = (0, _lodash.difference)(startingStreams, desiredStreams);
    const created = (0, _lodash.difference)(desiredStreams, startingStreams);
    const updated = (0, _lodash.intersection)(startingStreams, desiredStreams);
    return {
      created,
      updated,
      deleted
    };
  }
  get(name) {
    return this.streamsByName.get(name);
  }
  set(name, stream) {
    this.streamsByName.set(name, stream);
  }
  all() {
    return Array.from(this.streamsByName.values());
  }
  has(name) {
    return this.streamsByName.has(name);
  }
  toPrintable() {
    // Drop all references to the dependencies since they cannot be JSON.stringified
    return Object.fromEntries(Array.from(this.streamsByName).map(([key, stream]) => [key, stream.toPrintable()]));
  }
}
exports.State = State;