"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.asTrees = asTrees;
exports.buildStreamRows = buildStreamRows;
exports.enrichStream = void 0;
exports.filterCollapsedStreamRows = filterCollapsedStreamRows;
exports.filterStreamsByQuery = filterStreamsByQuery;
exports.isParentName = isParentName;
exports.shouldComposeTree = shouldComposeTree;
var _streamsSchema = require("@kbn/streams-schema");
var _helpers = require("../data_management/stream_detail_lifecycle/helpers/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SORTABLE_FIELDS = ['nameSortKey', 'retentionMs'];
function isParentName(parent, descendant) {
  return parent !== descendant && descendant.startsWith(parent + '.');
}
function shouldComposeTree(sortField) {
  // Always allow tree mode for nameSortKey
  return !sortField || sortField === 'nameSortKey';
}

// Returns all streams that match the query or are ancestors of a match
function filterStreamsByQuery(streams, query) {
  if (!query) return streams;
  const lowerQuery = query.toLowerCase();
  const nameToStream = new Map();
  streams.forEach(s => nameToStream.set(s.stream.name, s));

  // Find all streams that match the query
  const matching = streams.filter(s => s.stream.name.toLowerCase().includes(lowerQuery));
  const resultSet = new Map();
  for (const stream of matching) {
    // Add the match
    resultSet.set(stream.stream.name, stream);
    // Add all ancestors
    const ancestors = (0, _streamsSchema.getAncestors)(stream.stream.name);
    for (let i = 0; i < ancestors.length; ++i) {
      const ancestor = nameToStream.get(ancestors[i]);
      if (ancestor) {
        resultSet.set(ancestors[i], ancestor);
      }
    }
  }
  return Array.from(resultSet.values());
}

// Filters out rows that are children of collapsed streams
function filterCollapsedStreamRows(rows, collapsedStreams, sortField) {
  if (!shouldComposeTree(sortField)) return rows;
  const result = [];
  for (const row of rows) {
    // If any ancestor is collapsed, skip this row
    const ancestors = (0, _streamsSchema.getAncestors)(row.stream.name);
    let skip = false;
    for (let i = 0; i < ancestors.length; ++i) {
      if (collapsedStreams.has(ancestors[i])) {
        skip = true;
        break;
      }
    }
    if (!skip) result.push(row);
  }
  return result;
}
function buildStreamRows(enrichedStreams, sortField, sortDirection) {
  const isAscending = sortDirection === 'asc';
  const compare = (a, b) => {
    const av = a[sortField];
    const bv = b[sortField];
    if (typeof av === 'string' && typeof bv === 'string') {
      return isAscending ? av.localeCompare(bv) : bv.localeCompare(av);
    }
    if (typeof av === 'number' && typeof bv === 'number') {
      return isAscending ? av - bv : bv - av;
    }
    return 0;
  };
  const result = [];
  const pushNode = (node, level, rootMeta) => {
    result.push({
      ...node,
      level,
      ...rootMeta
    });
    if (node.children) {
      node.children.sort(compare).forEach(child => pushNode(child, level + 1, rootMeta));
    }
  };
  [...enrichedStreams].sort(compare).forEach(root => {
    const rootMeta = {
      rootNameSortKey: root.nameSortKey,
      rootDocumentsCount: root.documentsCount,
      rootRetentionMs: root.retentionMs
    };
    pushNode(root, 0, rootMeta);
  });
  return result;
}
function asTrees(streams) {
  const trees = [];
  const sortedStreams = streams.slice().sort((a, b) => (0, _streamsSchema.getSegments)(a.stream.name).length - (0, _streamsSchema.getSegments)(b.stream.name).length);
  sortedStreams.forEach(streamDetail => {
    let currentTree = trees;
    let existingNode;
    while (existingNode = currentTree.find(node => isParentName(node.stream.name, streamDetail.stream.name))) {
      currentTree = existingNode.children;
    }
    if (!existingNode) {
      currentTree.push({
        ...streamDetail,
        children: []
      });
    }
  });
  return trees;
}
const enrichStream = node => {
  let retentionMs = 0;
  const lc = node.effective_lifecycle;
  if ((0, _streamsSchema.isDslLifecycle)(lc)) {
    retentionMs = lc.dsl.data_retention ? (0, _helpers.parseDurationInSeconds)(lc.dsl.data_retention) * 1000 : Number.POSITIVE_INFINITY;
  } else if ((0, _streamsSchema.isIlmLifecycle)(lc)) {
    retentionMs = Number.POSITIVE_INFINITY;
  }
  const nameSortKey = 'children' in node ? `${(0, _streamsSchema.getSegments)(node.stream.name).length}_${node.stream.name.toLowerCase()}` : node.stream.name;
  const children = 'children' in node ? node.children.map(enrichStream) : undefined;
  return {
    stream: node.stream,
    effective_lifecycle: node.effective_lifecycle,
    data_stream: node.data_stream,
    nameSortKey,
    documentsCount: 0,
    retentionMs,
    type: _streamsSchema.Streams.ClassicStream.Definition.is(node.stream) ? 'classic' : (0, _streamsSchema.isRootStreamDefinition)(node.stream) ? 'root' : 'wired',
    ...(children && {
      children
    })
  };
};
exports.enrichStream = enrichStream;