"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToFieldDefinition = convertToFieldDefinition;
exports.getAllFieldsInOrder = getAllFieldsInOrder;
exports.getFilterSimulationDocumentsFn = getFilterSimulationDocumentsFn;
exports.getMappedSchemaFields = getMappedSchemaFields;
exports.getSchemaFieldsFromSimulation = getSchemaFieldsFromSimulation;
exports.getSourceField = getSourceField;
exports.getTableColumns = getTableColumns;
exports.getUniqueDetectedFields = getUniqueDetectedFields;
exports.getUnmappedSchemaFields = getUnmappedSchemaFields;
exports.mapField = mapField;
exports.unmapField = unmapField;
var _lodash = require("lodash");
var _fields = require("@kbn/streams-schema/src/fields");
var _types = require("../../../schema_editor/types");
var _utils = require("../../../schema_editor/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getSourceField(processor) {
  const processorSourceField = (() => {
    switch (processor.action) {
      case 'append':
      case 'set':
        return processor.to;
      case 'rename':
      case 'grok':
      case 'dissect':
      case 'date':
        return processor.from;
      case 'manual_ingest_pipeline':
        return undefined;
      default:
        return undefined;
    }
  })();
  const trimmedSourceField = processorSourceField === null || processorSourceField === void 0 ? void 0 : processorSourceField.trim();
  return trimmedSourceField && trimmedSourceField.length > 0 ? trimmedSourceField : undefined;
}
function getUniqueDetectedFields(detectedFields = []) {
  return (0, _lodash.uniq)(detectedFields.map(field => field.name));
}
function getAllFieldsInOrder(previewRecords = [], detectedFields = []) {
  const fields = new Set();
  previewRecords.forEach(record => {
    Object.keys(record).forEach(key => {
      fields.add(key);
    });
  });
  const uniqueDetectedFields = getUniqueDetectedFields(detectedFields);
  const otherFields = Array.from(fields).filter(field => !uniqueDetectedFields.includes(field));
  return [...uniqueDetectedFields, ...otherFields.sort()];
}
function getTableColumns({
  currentProcessorSourceField,
  detectedFields = [],
  previewDocsFilter
}) {
  if (!currentProcessorSourceField) {
    return [];
  }
  if (['outcome_filter_failed', 'outcome_filter_skipped'].includes(previewDocsFilter)) {
    return [currentProcessorSourceField];
  }
  const uniqueDetectedFields = getUniqueDetectedFields(detectedFields);
  return (0, _lodash.uniq)([currentProcessorSourceField, ...uniqueDetectedFields]);
}
function getFilterSimulationDocumentsFn(filter) {
  switch (filter) {
    case 'outcome_filter_parsed':
      return doc => doc.status === 'parsed';
    case 'outcome_filter_partially_parsed':
      return doc => doc.status === 'partially_parsed';
    case 'outcome_filter_skipped':
      return doc => doc.status === 'skipped';
    case 'outcome_filter_failed':
      return doc => doc.status === 'failed';
    case 'outcome_filter_all':
    default:
      return _doc => true;
  }
}
function getSchemaFieldsFromSimulation(context) {
  if (!context.simulation) {
    return {
      detectedSchemaFields: context.detectedSchemaFields,
      detectedSchemaFieldsCache: context.detectedSchemaFieldsCache
    };
  }
  const detectedFields = context.simulation.detected_fields;
  const updatedCache = new Map(context.detectedSchemaFieldsCache);
  const streamName = context.streamName;
  const streamType = context.streamType;
  const schemaFields = detectedFields.map(field => {
    // Detected field already mapped by the user on previous simulation
    const cachedField = updatedCache.get(field.name);
    if (cachedField) {
      return cachedField;
    }

    // Detected field unmapped by default
    let fieldSchema = {
      status: 'unmapped',
      esType: field.esType,
      name: field.name,
      parent: streamName
    };

    // Add description if available
    if ('description' in field && field.description) {
      fieldSchema.description = field.description;
    }

    // Add source if available (this is the metadata source like 'ecs', 'otel', etc.)
    if ('source' in field && field.source) {
      fieldSchema.source = field.source;
    }

    // Field Mapping Decision Tree:
    // ┌─────────────────────────────────────────────────────────────────────────────────┐
    // │ Field Processing Priority (top to bottom):                                     │
    // │                                                                                 │
    // │ 1. Inherited Fields (field.from exists)                                       │
    // │    └► status: 'inherited', parent: field.from                                │
    // │                                                                                 │
    // │ 2. ES Fields (field.esType exists)                                            │
    // │    └► status: 'mapped', retain esType, streamSource by stream type           │
    // │                                                                                 │
    // │ 3. Metadata Fields (suggestedType + source exists)                            │
    // │    ├► Classic Stream:                                                         │
    // │    │   ├► OTEL Classic Stream (name matches /^logs-.*\.otel-/):              │
    // │    │   │   ├► OTEL fields → status: 'mapped', streamSource: 'template'      │
    // │    │   │   └► All others → ignored (unmapped)                                │
    // │    │   ├► Regular Classic Stream:                                            │
    // │    │   │   ├► ECS fields → status: 'mapped', streamSource: 'template'       │
    // │    │   │   └► All others → ignored (unmapped)                                │
    // │    ├► Wired Stream:                                                          │
    // │    │   ├► OTEL fields → status: 'mapped', streamSource: 'stream'            │
    // │    │   └► All others → ignored (unmapped)                                    │
    // │    └► Unknown Stream → ECS fields only                                       │
    // │                                                                                 │
    // │ 4. User-Defined Fields (field.type exists)                                   │
    // │    └► status: 'mapped', use provided type                                    │
    // │                                                                                 │
    // │ 5. Fallback → status: 'unmapped' (unmanaged/dynamic)                         │
    // └─────────────────────────────────────────────────────────────────────────────────┘
    // Detected field already inherited
    if ('from' in field) {
      fieldSchema = {
        ...field,
        status: 'inherited',
        parent: field.from
      };
    } else {
      const isOtelField = 'source' in field && field.source === 'otel';
      const isEcsField = 'source' in field && field.source === 'ecs';
      const hasMetadataSuggestion = 'suggestedType' in field && field.suggestedType && 'source' in field && field.source;
      const isOtelClassicStream = streamType === 'classic' && streamName.match(/^logs-.*\.otel-/);

      // Field has ES type (already mapped in Elasticsearch)
      if (Boolean(field.esType)) {
        // Check if the ES type is supported by our FieldDefinitionType schema
        const isSupportedType = _fields.FIELD_DEFINITION_TYPES.includes(field.esType);
        if (isSupportedType) {
          // ES field with supported type - create as mapped field
          fieldSchema = {
            ...fieldSchema,
            status: 'mapped',
            streamSource: streamType === 'wired' ? 'stream' : 'template',
            esType: field.esType,
            type: field.esType
          };
        } else {
          // ES field with unsupported type - create as unmapped field but still show as existing in UI
          fieldSchema = {
            ...fieldSchema,
            status: 'unmapped',
            streamSource: streamType === 'wired' ? 'stream' : 'template',
            esType: field.esType
            // No type field set for unsupported ES types
          };
        }
      }
      // Field has metadata suggestion but no ES type - only handle ECS and OTEL fields
      else if (hasMetadataSuggestion) {
        if (streamType === 'classic') {
          // Classic streams: Handle ECS fields by default, but OTEL fields for OTEL-specific streams
          // Check if this is an OTEL-specific classic stream (e.g., logs-*.otel-*)
          const shouldMapField = isOtelClassicStream && isOtelField || !isOtelClassicStream && isEcsField;
          if (shouldMapField) {
            fieldSchema = {
              ...fieldSchema,
              status: 'mapped',
              type: field.suggestedType,
              streamSource: 'template',
              // Clear esType for metadata fields to ensure proper categorization
              esType: undefined
            };
          }
          // All other metadata fields are ignored for classic streams
        } else if (streamType === 'wired') {
          // Wired streams: Only handle OTEL fields, ignore everything else
          if (isOtelField) {
            fieldSchema = {
              ...fieldSchema,
              status: 'mapped',
              type: field.suggestedType,
              streamSource: 'stream',
              // Clear esType for metadata fields to ensure proper categorization
              esType: undefined
            };
          }
        } else {
          // Unknown stream type - fall back to handling ECS fields only
          if (isEcsField) {
            fieldSchema = {
              ...fieldSchema,
              status: 'mapped',
              type: field.suggestedType,
              streamSource: 'template',
              // Clear esType for metadata fields to ensure proper categorization
              esType: undefined
            };
          }
        }
      }
      // Field with user-defined type
      else if ('type' in field && field.type) {
        fieldSchema = {
          ...field,
          status: 'mapped',
          parent: streamName,
          type: field.type
        };
      }
    }
    updatedCache.set(fieldSchema.name, fieldSchema);
    return fieldSchema;
  }).filter(Boolean);
  return {
    detectedSchemaFields: schemaFields.sort(compareFieldsByStatus),
    detectedSchemaFieldsCache: updatedCache
  };
}
const statusOrder = {
  inherited: 0,
  mapped: 1,
  unmapped: 2
};
const compareFieldsByStatus = (curr, next) => {
  return statusOrder[curr.status] - statusOrder[next.status];
};
function mapField(context, updatedField) {
  const updatedCache = new Map(context.detectedSchemaFieldsCache);
  const updatedFields = context.detectedSchemaFields.map(field => {
    if (field.name !== updatedField.name) return field;
    const schemaField = {
      ...updatedField,
      status: 'mapped'
    };
    updatedCache.set(schemaField.name, schemaField);
    return schemaField;
  });
  return {
    detectedSchemaFields: updatedFields,
    detectedSchemaFieldsCache: updatedCache
  };
}
function unmapField(context, fieldName) {
  const updatedCache = new Map(context.detectedSchemaFieldsCache);
  const updatedFields = context.detectedSchemaFields.map(field => {
    if (field.name !== fieldName) return field;
    const schemaField = {
      ...field,
      type: undefined,
      status: 'unmapped'
    };
    updatedCache.set(schemaField.name, schemaField);
    return schemaField;
  });
  return {
    detectedSchemaFields: updatedFields,
    detectedSchemaFieldsCache: updatedCache
  };
}
function getMappedSchemaFields(fields) {
  return fields.filter(_types.isSchemaFieldTyped).filter(field => field.status === 'mapped');
}
function getUnmappedSchemaFields(fields) {
  return fields.filter(field => field.status === 'unmapped');
}
function convertToFieldDefinition(fields) {
  return fields.reduce((mappedFields, field) => Object.assign(mappedFields, {
    [field.name]: (0, _utils.convertToFieldDefinitionConfig)(field)
  }), {});
}