"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDataCollectionFailureNofitier = createDataCollectionFailureNofitier;
exports.createDataCollectorActor = createDataCollectorActor;
var _i18n = require("@kbn/i18n");
var _xstate = require("xstate5");
var _esQuery = require("@kbn/es-query");
var _rxjs = require("rxjs");
var _common = require("@kbn/data-plugin/common");
var _lodash = require("lodash");
var _errors = require("../../../../../util/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates a data collector actor that fetches sample documents based on the data source type
 */
function createDataCollectorActor({
  data
}) {
  return (0, _xstate.fromObservable)(({
    input
  }) => {
    const {
      dataSource,
      streamName
    } = input;
    return getDataCollectorForDataSource(dataSource)({
      data,
      index: streamName
    });
  });
}

/**
 * Returns the appropriate data collector function based on the data source type
 */
function getDataCollectorForDataSource(dataSource) {
  if (dataSource.type === 'random-samples') {
    return args => collectKqlData(args);
  }
  if (dataSource.type === 'kql-samples') {
    return args => collectKqlData({
      ...args,
      ...(0, _lodash.pick)(dataSource, ['filters', 'query', 'timeRange'])
    });
  }
  if (dataSource.type === 'custom-samples') {
    return () => (0, _rxjs.of)(dataSource.documents);
  }
  return () => (0, _rxjs.of)([]);
}

/**
 * Core function to collect data using KQL
 */
function collectKqlData({
  data,
  ...searchParams
}) {
  const abortController = new AbortController();
  const params = buildSamplesSearchParams(searchParams);
  return new _rxjs.Observable(observer => {
    const subscription = data.search.search({
      params
    }, {
      abortSignal: abortController.signal
    }).pipe((0, _rxjs.filter)(isValidSearchResult), (0, _rxjs.map)(extractDocumentsFromResult)).subscribe(observer);
    return () => {
      abortController.abort();
      subscription.unsubscribe();
    };
  });
}

/**
 * Validates if the search result contains hits
 */
function isValidSearchResult(result) {
  var _result$rawResponse$h;
  return !(0, _common.isRunningResponse)(result) && ((_result$rawResponse$h = result.rawResponse.hits) === null || _result$rawResponse$h === void 0 ? void 0 : _result$rawResponse$h.hits) !== undefined;
}

/**
 * Extracts documents from search result
 */
function extractDocumentsFromResult(result) {
  return result.rawResponse.hits.hits.map(doc => doc._source);
}

/**
 * Builds search parameters for Elasticsearch query
 */
function buildSamplesSearchParams({
  filters,
  index,
  query,
  size = 100,
  timeRange
}) {
  const queryDefinition = (0, _esQuery.buildEsQuery)({
    title: index,
    fields: []
  }, query !== null && query !== void 0 ? query : [], filters !== null && filters !== void 0 ? filters : []);
  addTimeRangeToQuery(queryDefinition, timeRange);
  return {
    index,
    allow_no_indices: true,
    query: queryDefinition,
    sort: [{
      '@timestamp': {
        order: 'desc'
      }
    }],
    size,
    terminate_after: size,
    track_total_hits: false
  };
}

/**
 * Adds time range to the query definition if provided
 */
function addTimeRangeToQuery(queryDefinition, timeRange) {
  if (timeRange) {
    queryDefinition.bool.must.unshift({
      range: {
        '@timestamp': {
          gte: timeRange.from,
          lte: timeRange.to
        }
      }
    });
  }
}

/**
 * Creates a notifier for data collection failures
 */
function createDataCollectionFailureNofitier({
  toasts
}) {
  return params => {
    const event = params.event;
    const error = (0, _errors.getFormattedError)(event.error);
    toasts.addError(error, {
      title: _i18n.i18n.translate('xpack.streams.enrichment.dataSources.dataCollectionError', {
        defaultMessage: 'An issue occurred retrieving documents from the data source.'
      }),
      toastMessage: error.message
    });
  };
}