"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertUIStepsToDSL = exports.convertStepsForUI = exports.convertStepToUIDefinition = void 0;
var _eui = require("@elastic/eui");
var _streamlang = require("../../../types/streamlang");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createId = (0, _eui.htmlIdGenerator)();
const convertStepsForUI = dsl => {
  const result = [];
  function unnestSteps(steps, parentId = null) {
    for (const step of steps) {
      const stepWithUI = convertStepToUIDefinition(step, {
        parentId
      });

      // If this is a Where block with nested steps, unnest them.
      // Remove the steps property, as these will now become flattened items.
      if ((0, _streamlang.isWhereBlock)(step) && Array.isArray(step.where.steps)) {
        // Add the where block itself
        result.push(stepWithUI);
        // Recursively unnest children, passing the current id as parentId
        unnestSteps(step.where.steps, stepWithUI.customIdentifier);
      } else {
        // Add non-where steps
        result.push(stepWithUI);
      }
    }
  }
  unnestSteps(dsl.steps);
  return result;
};
exports.convertStepsForUI = convertStepsForUI;
const convertStepToUIDefinition = (step, options) => {
  const id = step.customIdentifier || createId();

  // If this is a where step, remove where.steps.
  // UI versions of the steps keep a flat array and work off parentId to represent hierarchy.
  if ((0, _streamlang.isWhereBlock)(step) && Array.isArray(step.where.steps)) {
    const {
      steps,
      ...whereWithoutSteps
    } = step.where;
    return {
      customIdentifier: id,
      parentId: options.parentId,
      ...step,
      where: whereWithoutSteps
    };
  }
  return {
    customIdentifier: id,
    parentId: options.parentId,
    ...step
  };
};
exports.convertStepToUIDefinition = convertStepToUIDefinition;
const convertUIStepsToDSL = (steps, stripCustomIdentifiers = true) => {
  const idToStep = {};
  const rootSteps = [];

  // Prepare all steps and ensure where.steps exists for where blocks
  for (const step of steps) {
    const {
      customIdentifier,
      parentId,
      ...rest
    } = step;
    const stepObj = {
      ...rest,
      customIdentifier
    };
    // Where block
    if ((0, _streamlang.isWhereBlock)(stepObj)) {
      // Ensure where is always present and has steps
      stepObj.where = {
        ...stepObj.where,
        steps: []
      };
      idToStep[customIdentifier] = {
        ...stepObj,
        parentId
      };
    } else {
      idToStep[customIdentifier] = {
        ...stepObj,
        parentId
      };
    }
  }

  // Assign children to their parents recursively
  for (const step of Object.values(idToStep)) {
    const {
      parentId
    } = step;
    if (parentId && idToStep[parentId]) {
      const parent = idToStep[parentId];
      if ((0, _streamlang.isWhereBlock)(parent)) {
        parent.where.steps.push(step);
      }
    } else {
      rootSteps.push(step);
    }
  }

  // Remove parentId from all steps for the final DSL
  function stripUIProperties(step, removeCustomIdentifiers) {
    if ((0, _streamlang.isWhereBlock)(step)) {
      const {
        parentId,
        customIdentifier,
        ...whereRest
      } = step;
      return {
        ...(removeCustomIdentifiers ? {
          ...whereRest
        } : {
          ...whereRest,
          customIdentifier
        }),
        where: {
          ...whereRest.where,
          steps: whereRest.where.steps.map(child => stripUIProperties(child, removeCustomIdentifiers))
        }
      };
    } else {
      const {
        parentId,
        customIdentifier,
        ...actionRest
      } = step;
      return removeCustomIdentifiers ? actionRest : {
        ...actionRest,
        customIdentifier
      };
    }
  }
  return {
    steps: rootSteps.map(child => stripUIProperties(child, stripCustomIdentifiers))
  };
};
exports.convertUIStepsToDSL = convertUIStepsToDSL;