"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertDateProcessorToESQL = convertDateProcessorToESQL;
var _esqlAst = require("@kbn/esql-ast");
var _condition_to_esql = require("../condition_to_esql");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function convertDateProcessorToESQL(processor) {
  const {
    from,
    to,
    formats,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    output_format,
    where
  } = processor;
  const fromColumn = _esqlAst.Builder.expression.column(from);
  // In case ES has mapped fromColumn as datetime, we need to convert to string first for DATE_PARSE
  const fromAsString = _esqlAst.Builder.expression.func.call('TO_STRING', [fromColumn]);
  const targetDateField = to !== null && to !== void 0 ? to : '@timestamp'; // As with Ingest Date Processor, default to @timestamp
  const toColumn = _esqlAst.Builder.expression.column(targetDateField);
  const resolvedInputFormats = formats.map(f => resolveCommonDatePresetsForESQL(f));
  const resolvedOutputFormat = output_format ? resolveCommonDatePresetsForESQL(output_format) : undefined;
  const dateParseExpressions = resolvedInputFormats.map(f => _esqlAst.Builder.expression.func.call('DATE_PARSE', [_esqlAst.Builder.expression.literal.string(f), fromAsString]));
  const coalesceDateParse = _esqlAst.Builder.expression.func.call('COALESCE', dateParseExpressions);
  let dateProcessorAssignment = resolvedOutputFormat ? _esqlAst.Builder.expression.func.call('DATE_FORMAT', [_esqlAst.Builder.expression.literal.string(resolvedOutputFormat), coalesceDateParse]) : coalesceDateParse;
  if (where) {
    const whereCondition = (0, _condition_to_esql.conditionToESQLAst)(where);
    dateProcessorAssignment = _esqlAst.Builder.expression.func.call('CASE', [whereCondition, dateProcessorAssignment, toColumn]);
  }
  return [_esqlAst.Builder.command({
    name: 'eval',
    args: [_esqlAst.Builder.expression.func.binary('=', [toColumn, dateProcessorAssignment])]
  })];
}
function resolveCommonDatePresetsForESQL(format) {
  switch (format) {
    case 'ISO8601':
      return "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
    case 'RFC1123':
      return 'EEE, dd MMM yyyy HH:mm:ss z';
    case 'YYYY-MM-DD':
      return 'yyyy-MM-dd';
    case 'COMMON_LOG':
      return 'dd/MMM/yyyy:HH:mm:ss Z';
    case 'UNIX':
      return 'epoch_second';
    case 'UNIX_MS':
      return 'epoch_millis';
    default:
      return format;
  }
}