"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchEsqlQuery = fetchEsqlQuery;
exports.generateLink = generateLink;
exports.getEsqlQuery = void 0;
var _common = require("@kbn/triggers-actions-ui-plugin/common");
var _server = require("@kbn/task-manager-plugin/server");
var _i18n = require("@kbn/i18n");
var _esqlUtils = require("@kbn/esql-utils");
var _common2 = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchEsqlQuery({
  ruleId,
  alertLimit,
  params,
  services,
  spacePrefix,
  dateStart,
  dateEnd,
  sourceFields
}) {
  var _response$is_partial;
  const {
    logger,
    scopedClusterClient,
    share,
    ruleResultService
  } = services;
  const discoverLocator = share.url.locators.get('DISCOVER_APP_LOCATOR');
  const esClient = scopedClusterClient.asCurrentUser;
  const query = getEsqlQuery(params, alertLimit, dateStart, dateEnd);
  logger.debug(() => `ES|QL query rule (${ruleId}) query: ${JSON.stringify(query)}`);
  let response;
  try {
    response = await esClient.transport.request({
      method: 'POST',
      path: '/_query',
      body: query
    });
  } catch (e) {
    var _e$message;
    if ((_e$message = e.message) !== null && _e$message !== void 0 && _e$message.includes('verification_exception')) {
      throw (0, _server.createTaskRunError)(e, _server.TaskErrorSource.USER);
    }
    throw e;
  }
  const isGroupAgg = (0, _common.isPerRowAggregation)(params.groupBy);
  const {
    results,
    duplicateAlertIds
  } = await (0, _common2.getEsqlQueryHits)(response, params.esqlQuery.esql, isGroupAgg);
  if (ruleResultService && duplicateAlertIds && duplicateAlertIds.size > 0) {
    const warning = `The query returned multiple rows with the same alert ID. There are duplicate results for alert IDs: ${Array.from(duplicateAlertIds).join('; ')}`;
    ruleResultService.addLastRunWarning(warning);
    ruleResultService.setLastRunOutcomeMessage(warning);
  }
  const isPartial = (_response$is_partial = response.is_partial) !== null && _response$is_partial !== void 0 ? _response$is_partial : false;
  if (ruleResultService && isPartial) {
    const warning = getPartialResultsWarning(response);
    ruleResultService.addLastRunWarning(warning);
    ruleResultService.setLastRunOutcomeMessage(warning);
  }
  const link = generateLink(params, discoverLocator, dateStart, dateEnd, spacePrefix);
  return {
    link,
    parsedResults: (0, _common.parseAggregationResults)({
      ...results,
      resultLimit: alertLimit,
      sourceFieldsParams: sourceFields,
      generateSourceFieldsFromHits: true
    }),
    index: null
  };
}
const getEsqlQuery = (params, alertLimit, dateStart, dateEnd) => {
  const rangeFilter = [{
    range: {
      [params.timeField]: {
        lte: dateEnd,
        gt: dateStart,
        format: 'strict_date_optional_time'
      }
    }
  }];
  const query = {
    query: alertLimit ? `${params.esqlQuery.esql} | limit ${alertLimit}` : params.esqlQuery.esql,
    filter: {
      bool: {
        filter: rangeFilter
      }
    },
    ...((0, _esqlUtils.hasStartEndParams)(params.esqlQuery.esql) ? {
      params: [{
        _tstart: dateStart
      }, {
        _tend: dateEnd
      }]
    } : {})
  };
  return query;
};
exports.getEsqlQuery = getEsqlQuery;
function generateLink(params, discoverLocator, dateStart, dateEnd, spacePrefix) {
  const redirectUrlParams = {
    timeRange: {
      from: dateStart,
      to: dateEnd
    },
    query: params.esqlQuery,
    isAlertResults: true
  };

  // use `lzCompress` flag for making the link readable during debugging/testing
  // const redirectUrl = discoverLocator!.getRedirectUrl(redirectUrlParams, { lzCompress: false });
  const redirectUrl = discoverLocator.getRedirectUrl(redirectUrlParams, {
    spaceId: spacePrefix
  });
  return redirectUrl;
}
function getPartialResultsWarning(response) {
  var _response$_clusters$d, _response$_clusters;
  const clusters = (_response$_clusters$d = response === null || response === void 0 ? void 0 : (_response$_clusters = response._clusters) === null || _response$_clusters === void 0 ? void 0 : _response$_clusters.details) !== null && _response$_clusters$d !== void 0 ? _response$_clusters$d : {};
  const shardFailures = [];
  for (const cluster of Object.keys(clusters)) {
    var _clusters$cluster$fai, _clusters$cluster;
    const failures = (_clusters$cluster$fai = (_clusters$cluster = clusters[cluster]) === null || _clusters$cluster === void 0 ? void 0 : _clusters$cluster.failures) !== null && _clusters$cluster$fai !== void 0 ? _clusters$cluster$fai : [];
    if (failures.length > 0) {
      shardFailures.push(...failures);
    }
  }
  return _i18n.i18n.translate('xpack.stackAlerts.esQuery.partialResultsWarning', {
    defaultMessage: shardFailures.length > 0 ? 'The query returned partial results. Some clusters may have been skipped due to timeouts or other issues. Failures: {failures}' : 'The query returned partial results. Some clusters may have been skipped due to timeouts or other issues.',
    values: {
      failures: JSON.stringify(shardFailures)
    }
  });
}