"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultSummaryClient = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _constants = require("../../common/constants");
var _services = require("../domain/services");
var _date_range = require("../domain/services/date_range");
var _utils = require("./utils");
var _get_slices_from_date_range = require("./utils/get_slices_from_date_range");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// This is called "SummaryClient" but is responsible for:
// - computing summary
// - formatting groupings
// - adding extra Meta parameter for synthetics

class DefaultSummaryClient {
  constructor(esClient, burnRatesClient) {
    this.esClient = esClient;
    this.burnRatesClient = burnRatesClient;
  }
  async computeSummary({
    slo,
    instanceId,
    remoteName
  }) {
    var _result$aggregations, _result$aggregations$, _result$aggregations$2, _result$aggregations$3, _result$aggregations$4, _source$slo;
    const dateRange = (0, _date_range.toDateRange)(slo.timeWindow);
    const isDefinedWithGroupBy = ![slo.groupBy].flat().includes(_sloSchema.ALL_VALUE);
    const hasInstanceId = instanceId !== _sloSchema.ALL_VALUE;
    const shouldIncludeInstanceIdFilter = isDefinedWithGroupBy && hasInstanceId;
    const instanceIdFilter = shouldIncludeInstanceIdFilter ? [{
      term: {
        'slo.instanceId': instanceId
      }
    }] : [];
    const result = await this.esClient.search({
      index: remoteName ? `${remoteName}:${_constants.SLI_DESTINATION_INDEX_PATTERN}` : _constants.SLI_DESTINATION_INDEX_PATTERN,
      size: 0,
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': slo.id
            }
          }, {
            term: {
              'slo.revision': slo.revision
            }
          }, {
            range: {
              '@timestamp': {
                gte: dateRange.from.toISOString(),
                lte: dateRange.to.toISOString()
              }
            }
          }, ...instanceIdFilter]
        }
      },
      aggs: {
        ...(shouldIncludeInstanceIdFilter && {
          last_doc: {
            top_hits: {
              sort: [{
                '@timestamp': {
                  order: 'desc'
                }
              }],
              _source: {
                includes: ['slo.groupings', 'monitor', 'observer', 'config_id']
              },
              size: 1
            }
          }
        }),
        ...buildAggs(slo)
      }
    });
    const source = (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : (_result$aggregations$ = _result$aggregations.last_doc) === null || _result$aggregations$ === void 0 ? void 0 : (_result$aggregations$2 = _result$aggregations$.hits) === null || _result$aggregations$2 === void 0 ? void 0 : (_result$aggregations$3 = _result$aggregations$2.hits) === null || _result$aggregations$3 === void 0 ? void 0 : (_result$aggregations$4 = _result$aggregations$3[0]) === null || _result$aggregations$4 === void 0 ? void 0 : _result$aggregations$4._source;
    const groupings = source === null || source === void 0 ? void 0 : (_source$slo = source.slo) === null || _source$slo === void 0 ? void 0 : _source$slo.groupings;
    const sliValue = computeSliValue(slo, dateRange, result.aggregations);
    const errorBudget = computeErrorBudget(slo, sliValue);
    const burnRates = await this.burnRatesClient.calculate(slo, instanceId !== null && instanceId !== void 0 ? instanceId : _sloSchema.ALL_VALUE, [{
      name: '5m',
      duration: new _sloSchema.Duration(5, _sloSchema.DurationUnit.Minute)
    }, {
      name: '1h',
      duration: new _sloSchema.Duration(1, _sloSchema.DurationUnit.Hour)
    }, {
      name: '1d',
      duration: new _sloSchema.Duration(1, _sloSchema.DurationUnit.Day)
    }], remoteName);
    return {
      summary: {
        sliValue,
        errorBudget,
        status: (0, _services.computeSummaryStatus)(slo.objective, sliValue, errorBudget),
        fiveMinuteBurnRate: getBurnRate('5m', burnRates),
        oneHourBurnRate: getBurnRate('1h', burnRates),
        oneDayBurnRate: getBurnRate('1d', burnRates)
      },
      groupings: groupings ? (0, _utils.getFlattenedGroupings)({
        groupBy: slo.groupBy,
        groupings
      }) : {},
      meta: getMetaFields(slo, source !== null && source !== void 0 ? source : {})
    };
  }
}
exports.DefaultSummaryClient = DefaultSummaryClient;
function buildAggs(slo) {
  return _sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) ? {
    good: {
      sum: {
        field: 'slo.isGoodSlice'
      }
    },
    total: {
      value_count: {
        field: 'slo.isGoodSlice'
      }
    }
  } : {
    good: {
      sum: {
        field: 'slo.numerator'
      }
    },
    total: {
      sum: {
        field: 'slo.denominator'
      }
    }
  };
}
function getMetaFields(slo, source) {
  var _source$monitor$id, _source$monitor, _source$observer$name, _source$observer, _source$config_id;
  const {
    indicator: {
      type
    }
  } = slo;
  switch (type) {
    case 'sli.synthetics.availability':
      return {
        synthetics: {
          monitorId: (_source$monitor$id = (_source$monitor = source.monitor) === null || _source$monitor === void 0 ? void 0 : _source$monitor.id) !== null && _source$monitor$id !== void 0 ? _source$monitor$id : '',
          locationId: (_source$observer$name = (_source$observer = source.observer) === null || _source$observer === void 0 ? void 0 : _source$observer.name) !== null && _source$observer$name !== void 0 ? _source$observer$name : '',
          configId: (_source$config_id = source.config_id) !== null && _source$config_id !== void 0 ? _source$config_id : ''
        }
      };
    default:
      return {};
  }
}
function computeSliValue(slo, dateRange, bucket) {
  var _bucket$good$value, _bucket$good, _bucket$total$value, _bucket$total;
  const good = (_bucket$good$value = bucket === null || bucket === void 0 ? void 0 : (_bucket$good = bucket.good) === null || _bucket$good === void 0 ? void 0 : _bucket$good.value) !== null && _bucket$good$value !== void 0 ? _bucket$good$value : 0;
  const total = (_bucket$total$value = bucket === null || bucket === void 0 ? void 0 : (_bucket$total = bucket.total) === null || _bucket$total === void 0 ? void 0 : _bucket$total.value) !== null && _bucket$total$value !== void 0 ? _bucket$total$value : 0;
  if (_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod)) {
    const totalSlices = (0, _get_slices_from_date_range.getSlicesFromDateRange)(dateRange, slo.objective.timesliceWindow);
    return (0, _services.computeSLI)(good, total, totalSlices);
  }
  return (0, _services.computeSLI)(good, total);
}
function computeErrorBudget(slo, sliValue) {
  const initialErrorBudget = 1 - slo.objective.target;
  const consumedErrorBudget = sliValue < 0 ? 0 : (1 - sliValue) / initialErrorBudget;
  return (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget, _sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow) && _sloSchema.occurrencesBudgetingMethodSchema.is(slo.budgetingMethod));
}
function getBurnRate(burnRateWindow, burnRates) {
  var _burnRates$find$burnR, _burnRates$find;
  return (_burnRates$find$burnR = (_burnRates$find = burnRates.find(({
    name
  }) => name === burnRateWindow)) === null || _burnRates$find === void 0 ? void 0 : _burnRates$find.burnRate) !== null && _burnRates$find$burnR !== void 0 ? _burnRates$find$burnR : 0;
}