"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServerOptions = getServerOptions;
var _default_validation_error_handler = require("./default_validation_error_handler");
var _get_listener = require("./get_listener");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const corsAllowedHeaders = ['Accept', 'Authorization', 'Content-Type', 'If-None-Match', 'kbn-xsrf'];

/**
 * Converts Kibana `HttpConfig` into `ServerOptions` that are accepted by the Hapi server.
 */
function getServerOptions(config, {
  configureTLS = true
} = {}) {
  const cors = config.cors.enabled ? {
    credentials: config.cors.allowCredentials,
    origin: config.cors.allowOrigin,
    headers: corsAllowedHeaders
  } : false;
  const options = {
    host: config.host,
    port: config.port,
    // manually configuring the listener
    // @ts-expect-error HAPI types only define http1/https listener, not http2
    listener: (0, _get_listener.getServerListener)(config, {
      configureTLS
    }),
    // must set to true when manually passing a TLS listener, false otherwise
    tls: configureTLS && config.ssl.enabled,
    routes: {
      cache: {
        privacy: 'private',
        otherwise: 'private, no-cache, no-store, must-revalidate'
      },
      cors,
      payload: {
        maxBytes: config.maxPayload.getValueInBytes(),
        timeout: config.payloadTimeout
      },
      validate: {
        failAction: _default_validation_error_handler.defaultValidationErrorHandler,
        options: {
          abortEarly: false
        }
      }
    },
    state: {
      strictHeader: false,
      isHttpOnly: true,
      isSameSite: false // necessary to allow using Kibana inside an iframe
    }
  };
  return options;
}