"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getExceptionsPreUpdateItemHandler = void 0;
var _artifacts = require("../../../../common/endpoint/service/artifacts");
var _errors = require("../validators/errors");
var _validators = require("../validators");
var _utils = require("../../../../common/endpoint/service/artifacts/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getExceptionsPreUpdateItemHandler = endpointAppContextService => {
  return async function ({
    data,
    context: {
      request,
      exceptionListClient
    }
  }) {
    if (data.namespaceType !== 'agnostic') {
      return data;
    }
    let isEndpointArtifact = false;
    let validatedItem = data;
    const currentSavedItem = await exceptionListClient.getExceptionListItem({
      id: data.id,
      itemId: data.itemId,
      namespaceType: data.namespaceType
    });

    // We don't want to `throw` here because we don't know for sure that the item is one we care about.
    // So we just return the data and the Lists plugin will likely error out because it can't find the item
    if (!currentSavedItem) {
      return data;
    }
    const listId = currentSavedItem.list_id;

    // Validate Trusted Applications
    if (_validators.TrustedAppValidator.isTrustedApp({
      listId
    })) {
      isEndpointArtifact = true;
      const trustedAppValidator = new _validators.TrustedAppValidator(endpointAppContextService, request);
      validatedItem = await trustedAppValidator.validatePreUpdateItem(data, currentSavedItem);
      trustedAppValidator.notifyFeatureUsage(data, 'TRUSTED_APP_BY_POLICY');
    }

    // Validate Trusted Devices
    if (_validators.TrustedDeviceValidator.isTrustedDevice({
      listId
    })) {
      isEndpointArtifact = true;
      const trustedDeviceValidator = new _validators.TrustedDeviceValidator(endpointAppContextService, request);
      validatedItem = await trustedDeviceValidator.validatePreUpdateItem(data, currentSavedItem);
      trustedDeviceValidator.notifyFeatureUsage(data, 'TRUSTED_DEVICE_BY_POLICY');
    }

    // Validate Event Filters
    if (_validators.EventFilterValidator.isEventFilter({
      listId
    })) {
      isEndpointArtifact = true;
      const eventFilterValidator = new _validators.EventFilterValidator(endpointAppContextService, request);
      validatedItem = await eventFilterValidator.validatePreUpdateItem(data, currentSavedItem);
      eventFilterValidator.notifyFeatureUsage(data, 'EVENT_FILTERS_BY_POLICY');
    }

    // Validate host isolation
    if (_validators.HostIsolationExceptionsValidator.isHostIsolationException({
      listId
    })) {
      isEndpointArtifact = true;
      const hostIsolationExceptionValidator = new _validators.HostIsolationExceptionsValidator(endpointAppContextService, request);
      validatedItem = await hostIsolationExceptionValidator.validatePreUpdateItem(data, currentSavedItem);
      hostIsolationExceptionValidator.notifyFeatureUsage(data, 'HOST_ISOLATION_EXCEPTION_BY_POLICY');
      hostIsolationExceptionValidator.notifyFeatureUsage(data, 'HOST_ISOLATION_EXCEPTION');
    }

    // Validate Blocklists
    if (_validators.BlocklistValidator.isBlocklist({
      listId
    })) {
      isEndpointArtifact = true;
      const blocklistValidator = new _validators.BlocklistValidator(endpointAppContextService, request);
      validatedItem = await blocklistValidator.validatePreUpdateItem(data, currentSavedItem);
      blocklistValidator.notifyFeatureUsage(data, 'BLOCKLIST_BY_POLICY');
    }

    // Validate Endpoint Exceptions
    if (_validators.EndpointExceptionsValidator.isEndpointException({
      listId
    })) {
      isEndpointArtifact = true;
      const endpointExceptionValidator = new _validators.EndpointExceptionsValidator(endpointAppContextService, request);
      validatedItem = await endpointExceptionValidator.validatePreUpdateItem(data, currentSavedItem);

      // If artifact does not have an assignment tag, then add it now. This is in preparation for
      // adding per-policy support to Endpoint Exceptions as well as to support space awareness
      if (!(0, _utils.hasGlobalOrPerPolicyTag)(validatedItem)) {
        var _validatedItem$tags;
        validatedItem.tags = (_validatedItem$tags = validatedItem.tags) !== null && _validatedItem$tags !== void 0 ? _validatedItem$tags : [];
        validatedItem.tags.push(_artifacts.GLOBAL_ARTIFACT_TAG);
      }
      endpointExceptionValidator.notifyFeatureUsage(data, 'ENDPOINT_EXCEPTIONS');
    }
    if (isEndpointArtifact && endpointAppContextService.experimentalFeatures.endpointManagementSpaceAwarenessEnabled) {
      if (!(0, _utils.hasArtifactOwnerSpaceId)(validatedItem)) {
        if (!request) {
          throw new _errors.EndpointArtifactExceptionValidationError(`Missing HTTP Request object`);
        }
        const spaceId = (await endpointAppContextService.getActiveSpace(request)).id;
        (0, _utils.setArtifactOwnerSpaceId)(validatedItem, spaceId);
      }
      return validatedItem;
    }
    return isEndpointArtifact ? validatedItem : data;
  };
};
exports.getExceptionsPreUpdateItemHandler = getExceptionsPreUpdateItemHandler;