"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryDiagnosticsTaskConfig = createTelemetryDiagnosticsTaskConfig;
var _helpers = require("../helpers");
var _types = require("../types");
var _filterlists = require("../filterlists");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetryDiagnosticsTaskConfig() {
  const taskName = 'Security Solution Telemetry Diagnostics task';
  const taskType = 'security:endpoint-diagnostics';
  return {
    type: taskType,
    title: taskName,
    interval: '5m',
    timeout: '4m',
    version: '1.1.0',
    getLastExecutionTime: _helpers.getPreviousDiagTaskTimestamp,
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('diagnostic'), mdc);
      const trace = taskMetricsService.start(taskType);
      log.debug('Running telemetry task');
      try {
        if (!taskExecutionPeriod.last) {
          throw new Error('last execution timestamp is required');
        }
        let alertCount = 0;
        for await (const alerts of receiver.fetchDiagnosticAlertsBatch(taskExecutionPeriod.last, taskExecutionPeriod.current)) {
          const processedAlerts = alerts.map(event => (0, _filterlists.copyAllowlistedFields)(_filterlists.filterList.endpointAlerts, event));
          if (alerts.length === 0) {
            log.debug('no diagnostic alerts retrieved');
            await taskMetricsService.end(trace);
            return alertCount;
          }
          alertCount += alerts.length;
          log.debug('Sending diagnostic alerts', {
            alerts_count: alerts.length
          });
          sender.sendAsync(_types.TelemetryChannel.ENDPOINT_ALERTS, processedAlerts);
        }
        await taskMetricsService.end(trace);
        return alertCount;
      } catch (err) {
        await taskMetricsService.end(trace, err);
        return 0;
      }
    }
  };
}