"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ensureActionRequestsIndexIsConfigured = void 0;
var _lodash = require("lodash");
var _stringify = require("../../../utils/stringify");
var _constants = require("../../../../../common/endpoint/constants");
var _errors = require("../clients/errors");
var _utils = require("../../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ensureActionRequestsIndexIsConfigured = async endpointService => {
  const logger = endpointService.createLogger('ensureActionRequestsIndexIsConfigured');
  const esClient = endpointService.getInternalEsClient();
  const isSpacesEnabled = endpointService.experimentalFeatures.endpointManagementSpaceAwarenessEnabled;
  const COMPONENT_TEMPLATE_NAME = '.logs-endpoint.actions@package';
  logger.debug(`Checking setup of index [${_constants.ENDPOINT_ACTIONS_INDEX}]`);
  await esClient.indices.createDataStream({
    name: _constants.ENDPOINT_ACTIONS_INDEX
  }).catch(error => {
    var _error$body, _error$body$error;
    // Ignore error if the index already exists
    if (((_error$body = error.body) === null || _error$body === void 0 ? void 0 : (_error$body$error = _error$body.error) === null || _error$body$error === void 0 ? void 0 : _error$body$error.type) !== 'resource_already_exists_exception') {
      throw new _errors.ResponseActionsClientError(`Attempt to create index [${_constants.ENDPOINT_ACTIONS_INDEX}] failed with: ${error.message}`, error);
    }
    logger.debug(`Index for [${_constants.ENDPOINT_ACTIONS_INDEX}] already exists`);
  });
  if (!isSpacesEnabled) {
    logger.debug(`Space awareness feature is disabled. Nothing to do.`);
    return;
  }
  logger.debug(`Checking field mappings for index [${_constants.ENDPOINT_ACTIONS_INDEX}] in support of space awareness`);
  const indexMapping = await esClient.indices.getMapping({
    index: _constants.ENDPOINT_ACTIONS_INDEX
  }).catch(_utils.catchAndWrapError);
  logger.debug(() => `Index [${_constants.ENDPOINT_ACTIONS_INDEX}] existing mappings: ${(0, _stringify.stringify)(indexMapping)}`);
  const newRootMappings = {
    originSpaceId: {
      type: 'keyword',
      ignore_above: 1024
    },
    tags: {
      type: 'keyword',
      ignore_above: 1024
    }
  };
  const newAgentPolicyMappings = {
    agentId: {
      type: 'keyword',
      ignore_above: 1024
    },
    elasticAgentId: {
      type: 'keyword',
      ignore_above: 1024
    },
    integrationPolicyId: {
      type: 'keyword',
      ignore_above: 1024
    },
    agentPolicyId: {
      type: 'keyword',
      ignore_above: 1024
    }
  };
  const needsMappingUpdates = Object.entries(indexMapping).some(([backingIndexName, backingIndexMapping]) => {
    if (!(0, _lodash.get)(backingIndexMapping, 'mappings.properties.originSpaceId') || !(0, _lodash.get)(backingIndexMapping, 'mappings.properties.tags') || !(0, _lodash.get)(backingIndexMapping, 'mappings.properties.agent.properties.policy')) {
      logger.debug(`DS Backing index [${backingIndexName}] missing required mappings`);
      return true;
    }
    return false;
  });
  if (needsMappingUpdates) {
    logger.debug(`adding mappings to index [${_constants.ENDPOINT_ACTIONS_INDEX}]`);
    await esClient.indices.putMapping({
      index: _constants.ENDPOINT_ACTIONS_INDEX,
      properties: {
        ...newRootMappings,
        agent: {
          properties: {
            policy: {
              properties: newAgentPolicyMappings
            }
          }
        }
      }
    }).catch(_utils.catchAndWrapError);
    logger.info(`Index [${_constants.ENDPOINT_ACTIONS_INDEX}] was updated with new field mappings in support of space awareness`);
  } else {
    logger.debug(`Nothing to do - index [${_constants.ENDPOINT_ACTIONS_INDEX}] already has required mappings in support of space awareness`);
  }
  const componentTemplateExists = await esClient.cluster.existsComponentTemplate({
    name: COMPONENT_TEMPLATE_NAME
  }).catch(_utils.catchAndWrapError);
  if (componentTemplateExists) {
    var _currentComponentTemp;
    logger.debug(`Checking component template [${COMPONENT_TEMPLATE_NAME}] to ensure it has required mappings`);
    const currentComponentTemplate = await esClient.cluster.getComponentTemplate({
      name: COMPONENT_TEMPLATE_NAME
    });
    logger.debug(() => `current component template: ${(0, _stringify.stringify)(currentComponentTemplate, 15)}`);
    const componentMappings = (_currentComponentTemp = currentComponentTemplate.component_templates[0]) === null || _currentComponentTemp === void 0 ? void 0 : _currentComponentTemp.component_template.template.mappings;
    if (componentMappings && componentMappings.properties && (!(0, _lodash.get)(componentMappings, 'properties.originSpaceId') || !(0, _lodash.get)(componentMappings, 'properties.tags') || !(0, _lodash.get)(componentMappings, 'properties.agent.properties.policy'))) {
      logger.debug(`Adding mappings to component template [${COMPONENT_TEMPLATE_NAME}]`);
      Object.assign(componentMappings.properties, newRootMappings);
      if (componentMappings.properties.agent && 'properties' in componentMappings.properties.agent) {
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        componentMappings.properties.agent.properties.policy = {
          properties: newAgentPolicyMappings
        };
      }
      logger.debug(() => `Updating component template [${COMPONENT_TEMPLATE_NAME}] with:${(0, _stringify.stringify)(currentComponentTemplate, 15)}`);
      await esClient.cluster.putComponentTemplate({
        name: COMPONENT_TEMPLATE_NAME,
        template: {
          mappings: componentMappings
        }
      }).catch(_utils.catchAndWrapError);
      logger.info(`Component template [${COMPONENT_TEMPLATE_NAME}] was updated with mappings in support of space awareness`);
    }
  } else {
    logger.debug(`Component template [${COMPONENT_TEMPLATE_NAME}] does not exist. Nothing to check`);
  }
  logger.debug(`Checking of index [${_constants.ENDPOINT_ACTIONS_INDEX}] done.`);
};
exports.ensureActionRequestsIndexIsConfigured = ensureActionRequestsIndexIsConfigured;