"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ENTITY_RISK_SCORE_TOOL_ID = exports.ENTITY_RISK_SCORE_TOOL_DESCRIPTION = exports.ENTITY_RISK_SCORE_TOOL = void 0;
var _zod = require("@kbn/zod");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _tools = require("@langchain/core/tools");
var _helpers = require("@kbn/elastic-assistant-plugin/server/lib/langchain/helpers");
var _common = require("../../../../common/api/entity_analytics/common/common.gen");
var _get_risk_score = require("../../../lib/entity_analytics/risk_score/get_risk_score");
var _types = require("../../../../common/entity_analytics/types");
var _common2 = require("../../../../common");
var _get_alert_by_id = require("./get_alert_by_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ENTITY_RISK_SCORE_TOOL_DESCRIPTION = exports.ENTITY_RISK_SCORE_TOOL_DESCRIPTION = `Call this for knowledge about the latest entity risk score and the inputs that contributed to the calculation (sorted by 'kibana.alert.risk_score') in the environment, or when answering questions about how critical or risky an entity is. When informing the risk score value for a entity you must use the normalized field 'calculated_score_norm'.`;
const ENTITY_RISK_SCORE_TOOL_ID = exports.ENTITY_RISK_SCORE_TOOL_ID = 'entity-risk-score-tool';

/**
 * Returns a tool for querying entity risk score, or null if the
 * request doesn't have all the required parameters.
 */
const ENTITY_RISK_SCORE_TOOL = exports.ENTITY_RISK_SCORE_TOOL = {
  id: ENTITY_RISK_SCORE_TOOL_ID,
  name: 'EntityRiskScoreTool',
  // note: this description is overwritten when `getTool` is called
  // local definitions exist ../elastic_assistant/server/lib/prompt/tool_prompts.ts
  // local definitions can be overwritten by security-ai-prompt integration definitions
  description: ENTITY_RISK_SCORE_TOOL_DESCRIPTION,
  sourceRegister: _common2.APP_UI_ID,
  isSupported: params => {
    const {
      alertsIndexPattern,
      request,
      size
    } = params;
    return (0, _helpers.requestHasRequiredAnonymizationParams)(request) && alertsIndexPattern != null && size != null && !(0, _elasticAssistantCommon.sizeIsOutOfRange)(size);
  },
  async getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      alertsIndexPattern,
      anonymizationFields = [],
      onNewReplacements,
      replacements,
      assistantContext,
      logger,
      esClient
    } = params;
    return (0, _tools.tool)(async input => {
      var _localReplacements$in, _latestRiskScore$inpu;
      const spaceId = assistantContext.getSpaceId();
      const getRiskScore = (0, _get_risk_score.createGetRiskScores)({
        logger,
        esClient,
        spaceId
      });
      const getAlerts = (0, _get_alert_by_id.createGetAlertsById)({
        esClient
      });
      const entityField = _types.EntityTypeToIdentifierField[input.identifier_type];
      if ((0, _elasticAssistantCommon.isDenied)({
        anonymizationFields,
        field: entityField
      })) {
        return `The field ${entityField} is denied by the anonymization settings and cannot be used to identify the entity. Please modify the anonymization settings and try again.`;
      }

      // Accumulate replacements locally so we can, for example use the same
      // replacement for a hostname when we see it in multiple alerts:
      let localReplacements = replacements !== null && replacements !== void 0 ? replacements : {};
      const localOnNewReplacements = newReplacements => {
        localReplacements = {
          ...localReplacements,
          ...newReplacements
        };
        onNewReplacements === null || onNewReplacements === void 0 ? void 0 : onNewReplacements(localReplacements); // invoke the callback with the latest replacements
      };
      const deAnonymizedIdentifier = (_localReplacements$in = localReplacements[input.identifier]) !== null && _localReplacements$in !== void 0 ? _localReplacements$in : input.identifier;
      const riskScore = await getRiskScore({
        entityType: input.identifier_type,
        entityIdentifier: deAnonymizedIdentifier,
        pagination: {
          querySize: 1,
          cursorStart: 0
        }
      });
      if (riskScore.length === 0) {
        return 'No risk score found for the specified entity.';
      }
      const latestRiskScore = riskScore[0];

      // fetch all alerts that contributed to the risk score to enhance the inputs
      const alertsById = await getAlerts({
        index: alertsIndexPattern,
        ids: (_latestRiskScore$inpu = latestRiskScore.inputs.map(i => i.id)) !== null && _latestRiskScore$inpu !== void 0 ? _latestRiskScore$inpu : [],
        anonymizationFields
      });
      const enhancedInputs = riskScore.flatMap(r => r.inputs.map(i => ({
        risk_score: i.risk_score,
        contribution_score: i.contribution_score,
        category: i.category,
        alert_contribution: (0, _elasticAssistantCommon.transformRawData)({
          anonymizationFields,
          currentReplacements: localReplacements,
          getAnonymizedValue: _elasticAssistantCommon.getAnonymizedValue,
          onNewReplacements: localOnNewReplacements,
          rawData: (0, _elasticAssistantCommon.getRawDataOrDefault)(alertsById[i.id])
        })
      })));
      const data = {
        ...latestRiskScore,
        inputs: enhancedInputs,
        id_value: input.identifier
      }; // Replace id_value for the anonymized identifier to avoid leaking user data

      return JSON.stringify({
        riskScore: data,
        replacements: localReplacements
      });
    }, {
      name: 'EntityRiskScoreTool',
      description: params.description || ENTITY_RISK_SCORE_TOOL_DESCRIPTION,
      schema: _zod.z.object({
        identifier_type: _common.IdentifierType,
        identifier: _zod.z.string().min(1).describe('The value that identifies the entity.')
      }),
      tags: ['entity-risk-score', 'entities']
    });
  }
};